/**
 * マウスの状態を保持します。
 * 初期状態ではx座標、y座標を保持しますが、マウスアニメーションに必要なプロパティは自由に追加してください。
 */
export interface MouseState {
    x: number;
    y: number;
    [key: string]: any;
}
/**
 * マウスの制御を行う {@link Mouse} クラスのオブジェクトです。
 *
 * `negl`ではマウスの位置情報を内部で保持しています。これはマウスに追従するアニメーションや、{@link meshUI メッシュへのインタラクション} のマウス座標、もしくはシェーダに渡る [uMouse](/reference/modules/#defaultuniforms)、[uHover](/reference/modules/#defaultuniforms) の座標に用いられます。
 */
declare let mouse: Mouse;
/**
 * マウスの制御を行います。
 *
 * このクラスは`negl`内部でインスタンス化され、{@link mouse}オブジェクトとして提供されます。
 *
 * **取得方法**
 * ```ts
 * import { mouse } from "../negl";
 * // or
 * window.negl.mouse
 * ```
 */
declare class Mouse {
    #private;
    /**
     * アニメーション用のマウス座標が存在する位置を保持します。
     *
     * **[アニメーション用のマウス座標]**
     * mouse オブジェクトではブラウザのデフォルトのマウスカーソルから少し遅れた位置を current に保持しています。これによって、マウスに追従する独自のスタイルを実装することができます。どの程度遅延させるかは [config.mouse.speed](/reference/interfaces/Config/#マウスに関する設定) プロパティで設定します。
     */
    current: MouseState;
    /**
     * ブラウザのマウスカーソルの存在する位置を保持します。
     */
    target: MouseState;
    /**
     * ブラウザのマウスカーソル位置にアニメーション用のマウス座標の位置の変化量を保持します。
     */
    delta: MouseState;
    /**
     * マウスカーソルの初期値を保持します。
     */
    initial: MouseState;
    /**
     * マウスの位置が更新された回数を保持します。
     */
    tick: number;
    /**
     * マウスのアニメーション座標をブラウザのデフォルトのカーソル位置にどの程度の速度で追従させるかを数値で保持します。
     * [config.mouse.speed](/reference/interfaces/Config/#マウスに関する設定) プロパティに設定された値がこのプロパティに設定されます。
     */
    speed: number;
    static init(): Mouse;
    private constructor();
    /**
     * マウス位置をクリップ座標で返します。
     * @returns
     */
    getClipPos(): {
        x: number;
        y: number;
    };
    /**
     * マウス位置をUV座標で返します。
     * @returns
     */
    getUvPos(): {
        x: number;
        y: number;
    };
    /**
     * マウス位置を画面中央を0とした際の任意のスケールの座標で返します。
     * @returns
     */
    getMapPos(width: number, height: number): {
        x: number;
        y: number;
    };
    reset(): void;
    /**
     * newTarget のプロパティを {@link Mouse.target} オブジェクトに上書きします。
     * @param newTarget
     */
    setTarget(newTarget: Partial<MouseState>): void;
    /**
     * マウスの座標の更新をスタートする際に使用します。
     */
    startTrackMousePos(): void;
    /**
     * マウス座標の更新をストップする際に使用します。
     */
    stopTrackMousePos(): void;
    /**
     * マウス座標が前回の位置から変更されている場合は `true`、それ以外は `false` を返します。
     * @returns
     */
    isUpdate(): boolean;
}
export { mouse, Mouse };
//# sourceMappingURL=mouse.d.ts.map