import { TransitionState } from '../component/transition';
/**
 * Page モジュールはページ毎の処理を実行する際に使用します。
 * [data-page](/reference/html/#data-page) に設定された値によって Page モジュールの**デフォルトエクスポートした関数が実行**されます。基本的な使い方は[こちら](/tutorial/js-by-page/)をご覧ください。
 *
 * ### Page モジュール使用時の注意点
 * #### 初回ローディングの際は引数は渡りません。
 * 画面の初回ローディングの際は export default 関数には引数は渡りません。そのため、import 文によって必要なモジュールをインポートして利用してください。
 *
 * ```js
 * // import 文で negl の機能を取得
 * import { world } from "../negl";
 *
 * // 画面の初回ローディングの際は引数は渡ってこない。
 * export default async function() {
 *   const someEffect = world.getObByEl(".some-effect");
 * }
 * ```
 *
 * #### Page モジュールでエクスポートした関数はページトランジション時にアクセス可能です。
 * negl にはリロードなしで画面遷移を行う[ページトランジション](/tutorial/transition/)機能が存在します。
 * このページトランジション内ではPageモジュールでエクスポートされた関数にアクセスすることができます。
 *
 * #### 他のページから index.html ページへトランジションする時
 *
 * ##### index.html
 * ```html
 * <!-- .page-container で home を指定-->
 * <div class="page-container" data-page="home"></div>
 * ```
 *
 * ##### page/home.js
 * ```js
 * // トランジション 次のページのOBインスタンス挿入後
 * hook.on(hook.T_BOTH_OB_EXIST, runPageDefault, { priority: 1000 });
 * async function runPageDefault(state) {
 *  await state.next.page?.default?.(state);
 * }
 * ```
 *
 * ##### index.js
 * ```js
 * // 次のページのOBインスタンス挿入後
 * hook.on(hook.T_BOTH_OB_EXIST, runPageDefault, { priority: 1000 });
 * async function runPageDefault(state) {
 *  await state.next.page?.default?.(state);
 * }
 * ```
 *
 * また、Page モジュールでエクスポート（export）しているメソッドはページトランジション時に [T_*フック](/reference/classes/Hook/#on) から実行できます。
 * *[current.page](/reference/interfaces/TransitionState/#current) と [next.page](/reference/interfaces/TransitionState/#next) に現在ページと次のページの Page モジュールが渡ってきますので、このプロパティからアクセスしてください。
 *
 */
interface Page {
    /**
     * デフォルトエクスポートした関数は画面の初回ロード時に実行されます。
     * @param args
     * @returns
     */
    default: (args?: TransitionState) => Promise<void> | void;
    /**
     * Page モジュールに追加したメソッドはページトランジション時に [T_*フック](/reference/classes/Hook/#on) から実行できます。
     */
    [key: string]: (args?: TransitionState) => Promise<void> | void;
}
export { Page };
//# sourceMappingURL=page.d.ts.map