/**
 * スクロール変化量保持オブジェクト
 */
export type ScrollDelta = {
    /**
     * 画面横方向のスクロール変化量（現状、非対応です。* 2023/9/29）
     */
    x: number;
    /**
     * 画面縦方向のスクロール変化量
     */
    y: number;
};
/**
 * スクロールの制御を行う {@link Scroller} クラスのオブジェクトです。
 */
declare let scroller: Scroller;
/**
 * スクロールの制御を行います。
 *
 * このクラスは`negl`内部でインスタンス化され、{@link scroller} オブジェクトとして提供されます。
 *
 * **取得方法**
 * ```ts
 * import { scroller } from "../negl";
 * // or
 * window.negl.scroller
 * ```
 */
declare class Scroller {
    #private;
    /**
     * スクロール中の時は `true`、それ以外は `false` が設定されます。
     */
    scrolling: boolean;
    /**
     * スクロールの変化量が渡ります。
     *
     * スクロールに合わせてスクロールの変化量を取得したい場合は {@link Hook.on hook.on} の SCROLL タイプで取得してください。
     *
     * ```js
     * hook.on(hook.SCROLL, (scroller) => {
     *   console.log(scroller.delta);
     *   // スクロール中はスクロールの変化量がログに出力されます。
     * })
     * ```
     */
    delta: ScrollDelta;
    protected deltaTmp: ScrollDelta;
    static init(): Scroller;
    constructor();
    setDelta(delta: {
        x: number;
        y: number;
    }): void;
    onScroll: () => void;
    /**
     * スクロールを無効化します。（スクロールできないようになります。）
     */
    disable(): void;
    /**
     * スクロールを有効化します。（スクロールできるようになります。）
     */
    enable(): void;
    /**
     * [scrollIntoView](https://developer.mozilla.org/ja/docs/Web/API/Element/scrollIntoView) メソッドと同等の機能を持ちます。
     * scrollIntoView() が呼び出された要素がユーザーに見えるところまで、要素の親コンテナーをスクロールします。
     * @param el
     * @param scrollIntoViewOptions [scrollIntoViewOptions](https://developer.mozilla.org/ja/docs/Web/API/Element/scrollIntoView)
     * ただし、behavior オプションは "auto" のみ許容します。
     */
    scrollIntoView(el: HTMLElement, scrollIntoViewOptions?: ScrollIntoViewOptions): Promise<void>;
    /**
     * #id までページ内スクロールを行います。
     * @param hash id属性
     */
    scrollToHash(hash: string): void;
    /**
     * ページの最上部から px 分スクロールを行います。
     * @param px 画面最上部からスクロールしたい量を px で指定します。
     */
    set scrollTop(px: number);
    /**
     * ページの最上部からのスクロール量を数値で取得します。
     * @returns ページの最上部からのスクロール量
     */
    get scrollTop(): number;
}
export { scroller, Scroller };
//# sourceMappingURL=scroller.d.ts.map