/**
 * three.js(WebGL)制御用ファイル
 */
import { Clock, Object3D, Camera, WebGLRenderer, Scene, PerspectiveCamera } from 'three';
import { Viewport } from '../helper';
import { Ob } from '../glsl/Ob';
import lilGUI from 'lil-gui';
/**
 * {@link Ob}オブジェクト、HTML要素、またはセレクタ文字列
 */
export type ObOrSelector = Ob | Element | string;
export type RenderAction = (world: World) => void;
export type WorldOptions = {
    canvas: HTMLCanvasElement;
    clearColor: string;
    clearAlpha: number;
};
/**
 * {@link World}クラスのオブジェクト
 *
 * シーンの初期化、カメラの配置、メッシュ（Obクラス）の管理、レンダリングなどの3D空間の制御とメッシュの制御を行います。
 * プロパティやメソッドの詳細は{@link World}クラスを参照してください。
 *
 * **取得方法**
 * ```ts
 * import { world } from "../negl";
 * // or
 * window.negl.world
 * ```
 */
declare let world: World;
/**
 * シーン、カメラ、メッシュ、Obオブジェクト、ポストプロセッシングなどの[three.js](https://threejs.org/)を使った3D空間の全体的な制御を行います。
 * このクラスの機能を使用する際は既にインスタンス化された{@link world}変数を使用してください。
 *
 * @since 1.0.0
 */
declare class World {
    #private;
    /**
     * Obオブジェクトの
     */
    os: Ob[];
    /**
     * レンダリング開始時点からのティック回数
     */
    tick: number;
    /**
     * {@link https://threejs.org/docs/#api/en/core/Clock THREE.Clock}オブジェクト
     */
    clock: Clock;
    /**
     * {@link https://threejs.org/docs/#api/en/core/Clock.getDelta THREE.Clock#getDelta}の戻り値
     */
    delta: number;
    /**
     * {@link https://threejs.org/docs/#api/en/core/Scene THREE.Scene}オブジェクト
     */
    scene: Scene;
    /**
     * {@link https://threejs.org/docs/#api/en/core/WebGLRenderer THREE.WebGLRenderer}オブジェクト
     */
    renderer: WebGLRenderer;
    /**
     * {@link https://threejs.org/docs/?q=Camera#api/en/cameras/Camera THREE.Camera}オブジェクト
     */
    camera: Camera;
    renderStart: () => void;
    static init(options: WorldOptions): Promise<World>;
    private constructor();
    afterInit(): Promise<void>;
    cacheCurrentOs(): void;
    initObs(scope?: Document): Promise<Ob[]>;
    initOb(el: HTMLElement): Promise<Ob>;
    /**
     * HTML文字列、またはDOM要素からObオブジェクトを作成します。
     * これまではHTML内にエフェクトの元となるHTMLを必ず記述する必要がありましたが、JSから作成したい場合にこのメソッドを使用します。
     *
     * ```js
     * const templateOb = await world.createOb(`
     *  <img data-webgl="template" data-interactive data-click-1="/detail.html" />
     * `)
     * ```
     *
     * @param elOrHtmlStr HTML文字列、またはDOM要素
     * @param options 以下のオプションを設定可能です。
     * @param options.updateStyle
     *        `true` HTML要素に`{ visibility: 'hidden'; position: 'absolute'; top: 0; }`が付与された状態でDOMツリーに挿入されます。<br/>
     *        `false` スタイルは更新せず、DOMツリーに挿入します。
     * @param options.parent createObによって作成されたDOMを挿入する親のDOMを指定します。指定しない場合は `config.$.globalContainer` の子要素としてDOMが挿入されます。
     */
    createOb(elOrHtmlStr: HTMLElement | string, options?: {
        updateStyle?: boolean;
        parent?: string | HTMLElement;
    }): Promise<Ob>;
    /**
     * Obオブジェクトをシーン内に追加します。
     *
     * **注意**
     * HTML内の`data-webgl`で指定したObオブジェクトや{@link World#createOb}で作成したObオブジェクトは自動的にシーンに追加されます。
     * @param o Obオブジェクト
     */
    addOb(o: Ob): void;
    /**
     * [廃止]
     * @deprecated {@link World#removeOb removeOb} を代わりに使用してください。
     */
    removeObj(o: Ob, dispose?: boolean): void;
    /**
     * Obオブジェクトをシーン内から削除します。
     * @param o Obオブジェクト
     * @param dispose `true` Obオブジェクトに紐づく`Mesh`を完全に削除します。<br/>
     *                `false` シーンからのみ除外します。
     */
    removeOb(o: Ob, dispose?: boolean): true | void;
    /**
     * 引数で渡されたObオブジェクト配列に含まれるメッシュの[visible](https://threejs.org/docs/?q=Mesh#api/en/core/Object3D)プロパティを全て`true`（表示）の状態にします。
     * @param os Obオブジェクト配列
     */
    displayMeshes(os: Ob[]): void;
    /**
     * 引数で渡されたObオブジェクト配列に含まれるメッシュの[visible](https://threejs.org/docs/?q=Mesh#api/en/core/Object3D)プロパティを全て`false`（非表示）の状態にします。
     * @param os Obオブジェクト配列
     */
    hideMeshes(os: Ob[]): void;
    /**
     * メッシュに含まれる全ての`Material`と`Geometry`を再起的に破棄（`dispose`）します。
     * @param mesh メッシュ
     */
    disposeMesh(mesh: Object3D): void;
    /**
     * セレクタ文字列、またはDOM要素からObオブジェクトを取得します。
     * @param obOrSelector セレクタ文字列、DOM要素
     * @returns DOMに紐づくObオブジェクト
     *
     * ```ts copy
     * const panelO = world.getObByEl('.panel-selector');
     * panelO.mesh.visible = false; // メッシュを非表示
     * ```
     */
    getObByEl(obOrSelector: ObOrSelector): Ob | undefined;
    /**
     * セレクタ文字列に一致する全てのDOM要素に紐づくObオブジェクトを含む配列を取得します。
     * @param selector セレクタ文字列
     * @returns DOMに紐づくObオブジェクト配列
     *
     * ```ts copy
     * const panelOs = world.getObByElAll('.panel-selector');
     * panelOs[0].mesh.visible = false; // 一つ目のメッシュを非表示
     * panelOs[1].mesh.position.x += 500; // 二つ目のメッシュの位置を右に`500`移動
     * ```
     */
    getObByElAll(selector: string): (Ob | undefined)[];
    /**
     * [廃止]
     * @deprecated {@link World#getObByEl getObByEl} を代わりに使用してください。
     */
    getObjByEl(obOrSelector: ObOrSelector): Ob | undefined;
    /**
     * [PerspectiveCamera](https://threejs.org/docs/?q=PerspectiveCamera#api/en/cameras/PerspectiveCamera)の作成を行います。
     * @internal
     * @param viewport
     */
    setupPerspectiveCamera(viewport: Viewport): PerspectiveCamera;
    adjustWorldPosition(viewport: Viewport, duration?: number): Promise<void>;
    /**
     * 画面幅が変更した際のカメラの`fov`と`aspect`を更新します。
     * @param viewport
     */
    updateCamera(viewport: Viewport): void;
    /**
     * canvasへのレンダリングの処理を実行します。
     * このメソッドは requestAnimationFrame でループ実行されます。
     *
     * また、render メソッド内では以下の処理により画面描写を行っています。
     *
     * ```js filename="画面描写処理"
     * this.renderer.render(this.scene, this.camera);
     * ```
     *
     * その為、もしEffectComposer等でパスを適用したい場合は config.world.autoRender を false に設定し、以下のようにして composer.render を実行してください。
     *
     * ```js
     * // デフォルトの renderer.render の実行を無効化
     * config.world.autoRender = false;
     *
     * composer = new EffectComposer(world.renderer);
     * const renderPass = new RenderPass(world.scene, world.camera);
     * composer.addPass(renderPass);
     *
     * // hook.RENDERに登録（requestAnimationFrame のループ内で実行されるようになる）
     * hook.on(
     *   hook.RENDER,
     *   () => {
     *     // composer.render を requestAnimationFrame のタイミングで実行
     *     composer.render();
     *   },
     *   // priority: -1000 で実行の優先度を下げる
     *   { priority: -1000 }
     * );
     * ```
     *
     * また、[Travelサイト](/portfolio/travel)では getComposer.js という composer を取得する関数をご用意しています。こちらを使えば、裏側の処理を気にせず、composer を使用することができます。
     *
     * **getComposer.js**は以下のフォルダに格納されています。
     * travel/src/scripts/parts/helper/getComposer.js
     */
    render(): void;
    /**
     * メッシュに紐づくObクラスを取得します。
     * @param targetMesh
     * @returns `targetMesh`を含むObクラス
     */
    getObByMesh(targetMesh: Object3D): Ob | undefined;
    containsMesh(mesh: Object3D, targetMesh: Object3D): boolean;
    /**
     * lil-guiのOrbitControlチェックボックスを追加します。
     * @internal
     * @param gui lilGUIオブジェクト
     * @param defaultVal `true` チェックボックスにデフォルトチェック。`false` チェックなし。
     */
    addOrbitControlGUI(gui: lilGUI, defaultVal?: boolean): void;
    /**
     * [OrbitControls](https://threejs.org/docs/?q=OrbitControl#examples/en/controls/OrbitControls)をシーンに追加します。
     * @internal
     */
    attachOrbitControl(): void;
    /**
     * [OrbitControls](https://threejs.org/docs/?q=OrbitControl#examples/en/controls/OrbitControls)をシーンから削除します。
     * @internal
     */
    detachOrbitControl(): void;
    /**
     * イベントのバインド
     * @internal
     */
    bindEvents(): void;
    /**
     * [廃止]
     * @deprecated {@link MeshUI#watch MeshUI#watch} を代わりに使用してください。
     */
    addRaycastingTarget(obOrSelector: ObOrSelector): void;
}
export { world, World };
//# sourceMappingURL=world.d.ts.map