/**
 * 汎用関数
 */
import { Vector4, Vector3, Object3D } from "three";
import { Viewport } from "./viewport";
/**
 * negl 内部で使用する関数を保持する {@link Utils} クラスです。
 */
declare class Utils {
    /**
     * タッチデバイスかどうかを判定します。
     *
     * @returns タッチデバイスの時、true。それ以外、false。
     */
    isTouchDevices(): boolean;
    /**
     * Lerp 関数
     * @param a 始点
     * @param b 終点
     * @param n 線形補間要素（0 ~ 1）
     * @param limit limit まで値が近づくと終点の値を返す
     * @returns 線形補間された値
     */
    lerp(a: number, b: number, n?: number, limit?: number): number;
    /**
     * テクスチャのアスペクト値の算出
     *    * テクスチャとHTML要素の大きさのアスペクト比からuResolutionを設定します。
     * @param {DOMRect} toRect 要素の矩形を渡します。
     * @param {HTMLImageElement | HTMLVideoElement} mediaRect 画像、または動画のオリジナルサイズを渡します。
     * @returns {Vector4} uResolution に設定する Vector4 オブジェクト
     * x: DOMの横幅
     * y: DOMの縦幅
     * z: 横幅に掛ける縦横比
     * w: 縦幅に掛ける縦横比
     */
    getResolutionUniform(toRect: {
        height: number;
        width: number;
    }, mediaRect?: {
        height: number;
        width: number;
    }): Vector4;
    /**
     * 対角線上に頂点を詰めた配列を返却します
     * @param hSeg 高さのセグメント数
     * @param wSeg 幅のセグメント数
     * @param getValue 頂点毎に返却する値
     * @param defaultValue デフォルト値
     * @returns 頂点データの１次元配列を返却します。
     */
    getDiagonalVertices(hSeg: number, wSeg: number, getValue: (prevValue: number, currentIndex: number) => number, defaultValue: number): number[];
    /**
     * １次元の配列をテーブル形式で出力します
     * @param targetMatrix コンソールログに出力対象の配列を渡します。
     * @param col 列数を指定します。
     * @param label テーブルのラベルを指定します。
     * @returns
     */
    printMat(targetMatrix: THREE.Matrix4 | THREE.Matrix3 | THREE.BufferAttribute | number[], col?: number, label?: string): void;
    /**
     * メッシュの向きを指定の方向に回転します
     * @param _mesh 回転したいメッシュ
     * @param originalDir 元の向き
     * @param targetDir 向けたい向き
     */
    pointTo(_mesh: Object3D, originalDir: Vector3, targetDir: Vector3): void;
    /**
     * サファリブラウザかどうかを判定します。
     * @returns サファリの時 true。それ以外、false。
     */
    isSafari(): boolean;
    /**
     * iphone、ipadかどうかを判定します。
     * @returns iphone、ipadの時 true。それ以外、false。
     */
    isIOS(): boolean;
    /**
     * ケバブケースからキャメルケースへ文字列を変換します
     */
    toCamelCase(s: string): string;
    /**
     * HTMLの座標をワールド座標に変換します。
     * @param rect 対象のHTMLのDOMRect
     * @param viewport Canvasのビューポート
     * @returns
     */
    getWorldPosition(rect: DOMRect, viewport: Viewport): {
        x: number;
        y: number;
    };
    /**
     * `{ href, target }` を含むオブジェクトを返します。
     * `data-click-*` の値を `split` した配列からURLを取得する際に利用しています。
     *
     * ```js
     * console.log(utils.getHrefObj('newTab,/detail.html'.split()));
     * // > { href="/detail.html", target="_blank" }
     *
     * console.log(utils.getHrefObj('/detail.html'.split()));
     * // > { href="/detail.html", target="_self" }
     * ```
     */
    getHrefObj(args?: string[]): {
        href: string;
        target: "_self" | "_blank";
    } | undefined;
    /**
     * [gsap.to](https://greensock.com/docs/v3/GSAP/gsap.to())の簡易版
     * ライセンスの問題と依存するライブラリを増やしたくなかったため、gsap.toの簡易版の関数を作成しました。
     *
     * 基本的には gsap.to と使い方は変わりませんが、ease や delay は使用できません。
     * また、gsap.to と異なり、utils.tween では gsap.to の overwrite オプションが常に true の状態の tween となります。
     * ```js
     * const progress = { value: 0, value2: 1 }
     * utils.tween(progress, {
     *  value: 1,   // progress.value は 1 秒かけて　0 -> 1 に遷移します。
     *  value2: 0,  // progress.value2 は 1 秒かけて　1 -> 0 に遷移します。
     *  duration: 1,
     *  onStart() { console.log('スタート時に実行されます。') },
     *  onUpdate(progress) { console.log('アップデート時に実行されます。') },
     *  onComplete() { console.log('終了時に実行されます。') },
     * })
     * ```
     */
    tween(targets: {
        [key: string]: any;
    }, options: {
        duration: number;
        limit?: number;
        onStart?: () => void;
        onUpdate?: (progress: number) => void;
        onComplete?: (value?: unknown) => void;
        [key: string]: number | Function | undefined;
    }): void;
}
/**
 * negl 内部で使用する関数を保持する{@link Utils} クラスのオブジェクトです。
 */
declare const utils: Utils;
export { utils, Utils };
//# sourceMappingURL=utils.d.ts.map