import { Ob } from "../glsl";
import { Page } from "../component";
/**
 * ページトランジション関連のフックに登録したコールバック関数に渡るオブジェクトのインターフェイスです。
 * 現在ページ、次のページ、ページトランジションの種類、ページトランジションの契機等の情報を保持します。
 *
 * **対象のフック**
 * T_NEXT_DOM_BEFORE
 * T_BOTH_DOM_EXIST
 * T_BOTH_OB_EXIST
 * T_END
 * T_ERROR
 */
interface TransitionState {
    /**
     * 現在のページの状態を保持します。
     */
    current: {
        /**
         * 現在のページの`config.$.pageContainer`で取得したDOM要素を保持します。
         */
        el: HTMLElement | null;
        /**
         * 現在のページの`data-page`属性の値を保持します。
         */
        pageType: string | null;
        /**
         * 現在のページの`data-page`に紐づく{@link Page}モジュールを保持します。
         */
        page: Page | null;
        /**
         * 現在のページのURLのパスを保持します。
         */
        url: string;
        /**
         * 現在のページの {@link Ob}オブジェクトの配列を保持します。
         */
        os: Ob[] | null;
        /**
         * URLに紐づくデータ保存用マップ。URLに対応する{@link Map}オブジェクトが取得されます。
         * トランジション内で使用する値の保持・取得を行いたい場合に使用してください。<br/>
         * 値を保持する場合： ```current.store.set(key, value)```<br/>
         * 値を取得する場合： ```current.store.get(key)```
         */
        store: Map<string, any>;
    };
    /**
     * 次のページの状態を保持します。
     * negl はページ遷移を検知すると次のページの HTML をフェッチで取得します。
     * その為、取得完了までは次のページの情報にはアクセス出来ないため、大部分の情報は T_NEXT_DOM_BEFORE フック以降のタイミングでアクセス可能となります。
     */
    next: {
        /**
         * 次のページの`config.$.pageContainer`で取得したDOM要素を保持します。<br/>
         * 使用可能タイミング：T_NEXT_DOM_BEFORE 以降のフック
         */
        el: HTMLElement | null;
        /**
         * 次のページの`data-page`属性の値を保持します。<br/>
         * 使用可能タイミング：T_NEXT_DOM_BEFORE 以降のフック
         */
        pageType: string | null;
        /**
         * 次のページの`data-page`から取得した{@link Page}モジュールを保持します。<br/>
         * 使用可能タイミング：T_NEXT_DOM_BEFORE 以降のフック
         */
        page: Page | null;
        /**
         * 次のページのURLのパスを保持します。<br/>
         * 使用可能タイミング：T_BEGIN 以降のフック
         */
        url: string;
        /**
         * 次のページの{@link Ob}オブジェクトの配列を保持します。<br/>
         * 使用可能タイミング：T_BOTH_OB_EXIST 以降のフック
         */
        os: Ob[];
        /**
         * URLに紐づくデータ保存用マップ。URLに対応する{@link Map}オブジェクトが取得されます。
         * トランジション内で使用する値の保持・取得を行いたい場合に使用してください。<br/>
         * 値を保持する場合： ```next.store.set(key, value)```<br/>
         * 値を取得する場合： ```next.store.get(key)```<br/>
         * 使用可能タイミング：T_BEGIN 以降のフック
         */
        store: Map<string, any>;
        /**
         * 次のページの`<head>`のDOM要素を保持します。<br/>
         * 使用可能タイミング：T_NEXT_DOM_BEFORE 以降のフック
         */
        head: HTMLHeadElement | null;
        /**
         * 次のページの`Document`オブジェクトを保持します。<br/>
         * 使用可能タイミング：T_NEXT_DOM_BEFORE 以降のフック
         */
        doc: Document | null;
    };
    /**
     * メッシュ遷移での画面遷移の際には画面遷移の契機となったメッシュが渡ります。
     */
    triggerMesh: THREE.Object3D;
    /**
     * 画面クリックに対応するEventオブジェクトが渡ります。
     */
    event: Event | object;
    /**
     * `data-transition`、または `config.transition.defaultType` に設定された値が渡ります。
     */
    transitionType: string;
    /**
     * 画面遷移を発生させたトリガータイプが文字列で渡ります。
     * `link` リンクのクリック
     * `mesh` メッシュクリック
     * `popstate` ブラウザバック、ブラウザフォワード
     */
    triggerType: string;
    /**
     * ページトランジション中に任意のオブジェクトを保持することができます。
     */
    [key: string]: any;
}
export type GoToArgs = {
    href: string;
    target?: "_self" | "_blank";
    event?: Event;
    triggerType: "link" | "popstate" | "mesh";
    mesh?: THREE.Object3D | undefined;
    type?: string;
};
export type TransitionOptions = Partial<{
    spaMode: boolean;
    cache: boolean;
    defaultType: string;
}>;
/**
 * ページ遷移制御を行う{@link Transition}クラスのオブジェクトです。
 *
 * `negl`では [config.transition.spaMode](/reference/interfaces/Config/) を `true` にすることで SPA としてアプリを動かすことができます。SPAモードにした場合はリンククリック時やメッシュクリック時の画面ロードがなくなります。
 */
declare let transition: Transition;
/**
 * SPAモードでのページ遷移制御を行います。
 *
 * このクラスは`negl`内部でインスタンス化され、{@link transition}オブジェクトとして提供されます。
 *
 * **取得方法**
 * ```ts
 * import { transition } from "../negl";
 * // or
 * window.negl.transition
 * ```
 *
 * [注意]
 * SPAモードでのページ遷移制御は {@link hook} にコールバック関数を登録することで実装できます。その為、このオブジェクト自体を使用する機会は極めて稀です。
 *
 */
declare class Transition {
    #private;
    running: boolean;
    state?: TransitionState;
    static init(options: TransitionOptions): Transition;
    constructor({ spaMode, cache: enableCache, defaultType, }?: TransitionOptions);
    /**
     *
     * @param pathes
     */
    addExcludePath(...pathes: string[]): void;
    popstate(event: Event): void;
    static TRANSITION_TYPE: {
        NON_SPA: number;
        SPA: number;
        HASH: number;
    };
    /**
     * href として渡されたページに遷移します。
     *
     * @param href 遷移先のURL、またはパス
     * @param target _self の時、現在アクティブのタブでページ遷移を行います。_blank の時、別タブで開きます。
     * @param type トランジションタイプ
     */
    go(href: string, target?: "_self" | "_blank", type?: string): void;
    goTo({ href, target, event, triggerType, mesh, type }: GoToArgs): void;
    pageTransition({ href: nextHref, event, triggerType, mesh, type, }: GoToArgs): Promise<void>;
    /**
     * 戻り値で返却されたトランジションタイプでページ遷移が行われます。
     * JSで動的にトランジションタイプを決定したい場合に使用してください。
     *
     * **注意**
     * この処理は T_BEGIN の実行前に実行されます。その為、この時点では次のページの情報は基本的に取得できません。もし、次のページの種類によってトランジションタイプを変えたい場合は次のページの url などから判定してください。
     *
     * ```js
     * transition.changeType((transitionType, state) => {
     *   const { current, next, triggerType } = state;
     *   // 現在ページのページタイプ（data-pageの値）が `home` の時
     *   if (current.pageType === "home") {
     *     return "transition-type-1";
     *   }
     *   // 次のページのURLに detail が含まれていない場合
     *   if (next.url.indexOf("detail") !== -1) {
     *     return "transition-type-2"
     *   }
     *   // ブラウザバックの時
     *   if (triggerType === "popstate") {
     *     return "transition-type-3";
     *   }
     *   // それ以外の時、トランジションタイプを変更しない。
     *   return transitionType;
     * });
     * ```
     *
     * @param callback
     */
    changeType(callback: (transitionType: string, state: TransitionState) => string | Promise<string>): void;
}
export { transition, TransitionState, Transition };
//# sourceMappingURL=transition.d.ts.map