/**
 * サイト全体の設定値
 */
import lilGUI from "lil-gui";
import { Page, Scroller } from "../component";
import { ResolveArgs } from "../helper/module-resolver";
import { Ob } from "../glsl";
export type RegistPageArgs = ResolveArgs<Page>;
export type RegistObjArgs = ResolveArgs<{
    new (): Ob;
} & typeof Ob>;
/**
 * {@link setup} 関数に渡すことの出来るオプションの一覧です。
 *
 * **利用例**
 * ```js
 * const { start } = setup({
 *  world: { clearAlpha: 0 },   // canvasの背景の透明度を0とする
 *  viewport: { cameraZ: 2000 } // カメラのposition.zを2000とする
 * })
 * ```
 *
 * ### セレクタの設定
 * | プロパティ | 初期値 | 説明 |
 * | ------------------------- | ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------- |
 * | config.$.canvas | "#canvas" | WebGLを描写するcanvas要素のセレクタ |
 * | config.$.globalContainer | "#global-container" | ページ全体を囲むHTML要素のセレクタ |
 * | config.$.pageContainer | ".page-container" | ページ毎に内容を切り替えるHTML要素のセレクタ |
 * ### data-*属性のプレフィクス設定
 * | プロパティ | 初期値 | 説明 |
 * | ------------------------- | ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------- |
 * | config.prefix.ob | "webgl" | data-{prefix.ob}="gray-pic" |
 * | config.prefix.tex | "tex" | data-{prefix.tex}-1="/img/sample.jpg" |
 * | config.prefix.page | "page" | data-{prefix.page}="home" |
 * | config.prefix.click | "click" | data-{prefix.click}-1="/sample.html" |
 * | config.prefix.transition | "transition" | data-{prefix.transition}="some-transition" |
 * | config.prefix.interactive | "interactive" | data-{prefix.interactive} |
 * ### イベントリスナのタイプ設定
 * ***基本変更不可**
 * | プロパティ | 初期値 | 説明 |
 * | ------------------------- | ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------- |
 * | config.event.resize | "resize" | addEventListener(event.resize, ...) |
 * | config.event.mouse | "mouse" | addEventListener(event.mouse + "move", ...)<br>addEventListener(event.mouse + "enter", ...)<br>addEventListener(event.mouse + "leave", ...) |
 * | config.event.click | "click" | addEventListener(event.mouse + "click", ...) |
 * ### ページトランジション設定
 * | プロパティ | 初期値 | 説明 |
 * | ------------------------- | ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------- |
 * | config.transition.spaMode | false | `true` 画面ロードなしで他のページへ画面遷移（SPA）<br>`false` 画面ロードありで他の ージへ画面遷移 |
 * | config.transition.cache | true | `true` spaMode使用時、訪問したページの状態をメモリに保持。次回遷移時にページ表示速度が高速化します。<br/>`false` 毎回サーバにデータをフェッチ<br/>***この機能はベータ機能です。** |
 * | config.transition.defaultType | "distortion" | data-transition属性が設定されていない時のトランジションタイプ |
 * ### scroller設定
 * | プロパティ | 初期値 | 説明 |
 * | ------------------------- | ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------- |
 * | config.scroller.damping | 0.5 |  0~1 で メッシュのスクロール速度を決定します。|
 * ### リサイズやカメラの情報設定
 * | プロパティ | 初期値 | 説明 |
 * | ------------------------- | ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------- |
 * | config.viewport.autoresize | true | `true` リサイズイベント時に自動的にカメラやメッシュのリサイズ処理を実行 <br>`false` 自動的にカメラやメッシュのリサイズ処理を行わない。 |
 * | config.viewport.resizeDuration | 1 | メッシュのリサイズ処理の遷移に掛ける時間を秒単位で指定します。 |
 * | config.viewport.delay | 500 | 最後のリサイズイベントが発火された時点から何ms（ミリ秒）後に [RESIZE_DELAY](/reference/classes/Hook.md#on) を発火させるかを指定します。 |
 * | config.viewport.devicePixelRatio | 1 | WebGLRenderer の[ピクセル比](https://threejs.org/docs/#api/en/renderers/WebGLRenderer.setPixelRatio)に使用します。 |
 * | config.viewport.cameraZ | 2000 | [PerspectiveCamera](https://threejs.org/docs/#api/en/cameras/PerspectiveCamera)のz軸ポジション |
 * | config.viewport.near | 1500 | [PerspectiveCamera](https://threejs.org/docs/#api/en/cameras/PerspectiveCamera)のnear |
 * | config.viewport.far | 4000 | [PerspectiveCamera](https://threejs.org/docs/#api/en/cameras/PerspectiveCamera)のfar |
 * ### マウスに関する設定
 * | プロパティ | 初期値 | 説明 |
 * | ------------------------- | ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------- |
 * | config.mouse.speed | 0.2 | マウスの追従スピード（0.1 ~ 1.0）<br/>大きくするほど、追従スピードが上がります。 |
 *
 * ### three.jsに関する設定
 * | プロパティ | 初期値 | 説明 |
 * | ------------------------- | ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------- |
 * | config.world.autoRender | true | `true` 自動的にフレーム毎にシーンのレンダリング（renderer.render(scene, camera)の実行）を実行。<br/>`false` 自動的にシーンのレダリングを実行しない。<br/>EffectComposer等を使用する場合はこちらの設定値を使い、ご自身で composer.render() を実行するようにしてください。 |
 * | config.world.clearColor | "#000000" | canvas要素の背景色<br/>[setClearColor](https://threejs.org/docs/?q=webglrend#api/en/renderers/WebGLRenderer.setClearColor) の第一引数（color）に使用されます。 |
 * | config.world.clearAlpha | 1 | canvas要素の背景の透明度<br/>[setClearColor](https://threejs.org/docs/?q=webglrend#api/en/renderers/WebGLRenderer.setClearColor) の第二引数（alpha）に使用されます。 |
 *
 * 基本的にthree.jsの設定値を変更したい場合は {@link World} オブジェクトに格納されている renderer や scene を取得し、直接変更してください。
 *
 * ```js filename="three.jsの設定を変更"
 * const { world } = negl;
 * // シーンのトーンを調整
 * world.renderer.toneMapping = ACESFilmicToneMapping;
 * world.renderer.toneMappingExposure = 0.5;
 *
 * // Fogの追加
 * world.scene.fog = new Fog(0xc7d4d9, 0, 2500);
 * ```
 *
 * ### override設定
 * | プロパティ | 初期値 | 説明 |
 * | ------------------------- | ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------- |
 * | config.override.scroller | - | Scroller クラスを継承した別のクラスを設定します。 |                                                                    |
 *
 * ### addon設定
 * | プロパティ | 初期値 | 説明 |
 * | ------------------------- | ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------- |
 * | config.addon.lilGUI | - | [lilGUI](https://github.com/georgealways/lil-gui) を使用する場合に設定してください。<br/>使用の際は、併せて config.debug を true に設定してください。 |                                                                    |
 *
 * ### デバッグ設定
 * | プロパティ | 初期値 | 説明 |
 * | ------------------------- | ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------- |
 * | config.debug | true | `true` 開発モードで実行。（lilGUIの表示）<br>`false` 本番モードで実行。 |                                                                    |
 */
interface Config {
    resolver: {
        ob: RegistObjArgs;
        page: RegistPageArgs;
    };
    $: {
        canvas: string;
        globalContainer: string;
        pageContainer: string;
    };
    event: {
        resize: string;
        mouse: string;
        click: string;
    };
    prefix: {
        ob: string;
        tex: string;
        page: string;
        click: string;
        transition: string;
        interactive: string;
    };
    viewport: {
        autoresize: boolean;
        resizeDuration: number;
        delay: number;
        devicePixelRatio: number;
        cameraZ: number;
        near: number;
        far: number;
    };
    world: {
        autoRender: boolean;
        clearColor: string;
        clearAlpha: number;
    };
    meshUI: {
        shouldObserve?: () => boolean;
    };
    transition: {
        spaMode: boolean;
        cache: boolean;
        defaultType: string;
    };
    scroller: {
        damping: number;
    };
    gui: {
        guiOpen: boolean;
        folderOpen: boolean;
    };
    addon: {
        lilGUI?: typeof lilGUI;
    };
    mouse: {
        speed: number;
    };
    debug: boolean;
    override?: {
        scroller: typeof Scroller;
    };
}
/**
 * ライブラリ全体の設定を保持します。
 * 設定の上書きは {@link setup} 関数の実行時に行うことができます。
 *
 * ```ts copy filename="index.js" {4,5}
 * import { setup } from "../negl";
 *
 * function init() {
 *  const myConf = { world: { clearColor: "#ff0000" }} // 背景色を赤色に設定
 *  const { start } = await setup(myConf);
 *  const negl = await start();
 *  negl.renderStart();
 *  // 背景が赤色で表示される。
 * }
 * init();
 * ```
 *
 */
declare const config: Config;
declare const mergeConfig: (_config: Partial<Config>) => Config;
export { config, Config, mergeConfig };
//# sourceMappingURL=config.d.ts.map