import "../styles/style.scss";
import lilGUI from "lil-gui";
import { setup } from "negl";
import { definePerformanceMode } from "#/parts/helper/detectGPU";
import { performanceMonitor } from "#/parts/helper/performance-monitor";
import {
  letsBegin,
  progressCounter,
} from "#/parts/loading/loader-basic1/example";

// scroller
import { SmoothScroller } from "#/parts/helper/scroll/SmoothScroller";
import {
  init,
  addActions,
  mountAnimations,
} from "#/parts/helper/scroll/scroll-trigger";
import * as scrollActions from "#/parts/helper/scroll/scroll-animation";

// transition
import { registTransitionMenu } from "#/parts/transition/transition-menu";
import { registTransitionBase } from "#/parts/transition/transition-base";
import { registTransitionDistortion } from "#/parts/transition/transition-distortion";

// Mouse animation
import initMouseCircle from "#/parts/mouse/mouse-circle";
// メニューの開閉. Menu open/close
import { menu } from "#/parts/menu/three-bar-menu";

const enableDebugMode = (debug) => {
  return debug && import.meta.env.DEV;
};

// ページの読み込み
// Page loading
const pageResolver = (type) => import(`./page/${type}.js`);

// OBの読み込み
// Loading OB
const obResolver = async (type) => {
  const module = await import(`./parts/glsl/${type}/index.js`);

  return module.default;
};

(async () => {
  // パフォーマンスモードの初期化
  // neglの設定 *必ず一番最初に呼び出す。
  // Initialization of performance mode
  // negl settings *Always call first.
  await definePerformanceMode(2, 35);

  const debug = enableDebugMode(1);

  const { start, hook } = setup({
    resolver: {
      page: pageResolver,
      ob: obResolver,
    },
    world: { clearAlpha: 0 },
    viewport: {
      near: 1500,
      far: 4000,
      cameraZ: 2000,
    },
    transition: { defaultType: "distortion", spaMode: true },
    addon: { lilGUI },
    override: { scroller: SmoothScroller },
    debug,
  });

  // スクロールに伴うアニメーションの初期化;
  // Initialization of animations triggered by scrolling;
  hook.on(hook.BEFORE_PAGE_INIT, ({ scroller }) => {
    init(scroller);
  });

  /**
   * neglの初期化
   * 以降、negl内の全てのオブジェクトにアクセス可能。
   *
   * Initialization of negl
   * From now on, all objects within negl are accessible.
   */
  const negl = await start(progressCounter);
  const { INode, world, config, scroller } = negl;

  // ページトランジションの追加
  // Adding page transitions
  registTransitionMenu(hook);
  registTransitionDistortion({ hook, INode, config, world });
  registTransitionBase({ hook, scroller });
  hook.on(
    hook.T_BOTH_DOM_EXIST,
    () => {
      // スクロールトリガーのせいで画面中央でページに遷移すると、背景色がちらつく時があるため、画面最上部を設定
      // Due to the scroll trigger, setting the screen to the top to prevent background flickering when transitioning pages in the center of the screen
      scroller.scrollTop = 0;
      // transition-base.js の scrollToPos より後に実行
      // Execute after scrollToPos in transition-base.js
    },
    { priority: 899 }
  );

  // 背景のモコモコのエフェクトを作成
  // Create the fluffy background effect
  const fresnel = await world.createOb(`
  <div
    class="fresnel"
    data-webgl="fresnel-bg"
    data-tex-1="/movie/fresnel_back.webm"
    data-tex-2="/movie/fresnel_back.mov"
  ></div>
`);
  // 背景のモコモコのz位置を-1000として他のエフェクトの背後に配置
  // Set the z-position of the fluffy background to -1000 and place it behind other effects
  fresnel.mesh.position.z = -1000;
  fresnel.fixed = true;

  // gsap/ScrollTriggerへの登録
  // Register with gsap/ScrollTrigger
  addActions(scrollActions);
  // data-scroll-trigger を ScrollTrigger に登録
  // Register data-scroll-trigger with ScrollTrigger
  mountAnimations();

  // menuの初期化（ライブラリではなく、あくまでユーザー独自の機能として扱う。）
  // Initialization of the menu (treated as a user's own feature, not a library.)
  menu.init({ world, scroller });

  // マウスアニメーションの初期化
  // Initialization of mouse animation
  const mouseLoad = initMouseCircle(negl);

  /**
   * NOTE requestAnimationFrameによるシーンのレンダリング開始！
   * シーンを表示する直前に描写を開始
   *
   * NOTE Start rendering the scene with requestAnimationFrame!
   * Begin drawing just before displaying the scene
   */
  negl.renderStart();

  // パフォーマンスの測定
  // Performance measurement
  performanceMonitor(negl);

  /**
   * ローディング完了時のアニメーションの開始
   * あくまでユーザーの独自関数のため、ライブラリの機能ではありません。
   *
   * Start of animation upon loading completion
   * It is purely a user's own function and not a feature of the library.
   */
  await letsBegin();

  // マウス表示
  // Display mouse
  mouseLoad();
})();
