/**
 * ホーム画面（index.html）用制御ファイル
 * 概要：data-page="home"をキーに実行される
 *
 * Control file for the home screen (index.html)
 * Overview: Executed with data-page="home" as the key
 */
import initDistortionPass from "#/parts/glsl/distortion/pass";
import { isLowPerformanceMode } from "#/parts/helper/detectGPU";
import { addLoadingAnimation } from "#/parts/loading/loader-basic1/example";
import {
  mountNavBtnHandler,
  mountScrollHandler,
} from "#/parts/slider/slide-handler";
import { INode, world, meshUI } from "negl";

let _goTo = null,
  _pass = null;
export default async function () {
  // .panel__mediaのマウス情報（uMouse、uHover）を監視対象へ
  // Add mouse information (uMouse, uHover) of .panel__media to the watch list
  const planeEls = INode.qsAll(".panel__media");
  planeEls.forEach((planeEl) => meshUI.watch(planeEl));

  // ローディング用のパスを初期化
  // Initialize the path for loading
  _pass = await initDistortionPass({ world });

  // ファーストビューのナビゲーションボタンにクリックイベントを追加
  // Add click events to the navigation buttons of the first view
  const { goTo } = mountNavBtnHandler({
    sliderSelector: ".fv__slider",
    prevBtnSelector: ".fv__btn.prev",
    nextBtnSelector: ".fv__btn.next",
    textSelector: ".fv__text-shader",
  });

  _goTo = goTo;

  // SKILLセクションのスライダをスクロールに連動
  // Link the slider of the SKILL section to scroll
  mountScrollHandler({
    sliderSelector: ".skill__slider",
    triggerSelector: ".skill",
    textSelector: ".skill__ul",
  });

  // ファーストビューのテキストのエフェクトを取得
  // Retrieve the text effect of the first view
  const fvText = world.getObByEl(".fv__text-shader");
  if (fvText) {
    // ファーストビューのテキストのエフェクトのz位置を200としてスライダと被らないようにする
    // Set the z position of the first view's text effect to 200 to avoid overlapping with the slider
    fvText.mesh.position.z = 200;
  }

  // VISIONセクション内のエフェクトを取得
  // Retrieve the effects within the VISION section
  const raymarching = world.getObByEl(".vision__raymarching");
  const fallback = world.getObByEl(".vision__fallback");

  if (isLowPerformanceMode()) {
    // ローパフォーマンスモードの時、高負荷のレイマーチングを削除
    // In low performance mode, remove the high-load raymarching
    world.removeOb(raymarching);
    // .vision__fallbackのエフェクトにマウス入力を連携
    // Link mouse input to the effect of .vision__fallback
    meshUI.watch(".vision__fallback");
  } else {
    // 高パフォーマンスモードの時、フォールバック用の.vision__fallbackを削除
    // In high performance mode, remove the fallback .vision__fallback
    world.removeOb(fallback);
    // レイマーチングのエフェクトにマウス入力を連携
    // Link mouse input to the raymarching effect
    meshUI.watch(".vision__raymarching");
  }

  // ローディングアニメーションを追加
  // Add loading animation
  addLoadingAnimation(loadAnimation);
}

// ローディングアニメーション完了時の処理
// Process after loading animation completion
function loadAnimation(tl) {
  // ファーストビューのスライドを一回転する
  // Rotate the first view slide once
  _goTo(5);
  // ディストーションのパスをアニメーション
  // Animate the distortion path
  const distorionProgress = { value: 0 };
  tl.to(distorionProgress, {
    value: 1,
    duration: 0.3,
    // パスを追加する
    // Add the path
    onStart: _pass.addPass,
    onUpdate() {
      // プログレスの値を更新する
      // Update the progress value
      _pass.setProgress(distorionProgress.value);
    },
    // 処理が完了すれば不要なので、パスを削除する
    // Remove the path once the process is complete as it is no longer needed
    onComplete: _pass.removePass,
  });
}
