import { Scroller, INode, config, hook } from "negl";
import SmoothScrollbar from "smooth-scrollbar";
import { Data2d } from "smooth-scrollbar/interfaces";
import { scroller } from "../../../../src/component/scroller";

/**
 * smooth-scrollbar を用いてスクロールの制御を行います。
 * Control scrolling using smooth-scrollbar.
 *
 * 既存の Scroller クラスの代わりにこのクラスを使用するためには config.override.scroller にクラスを設定します。
 * To use this class instead of the existing Scroller class, set the class in config.override.scroller.
 *
 * ```js
 * import SmoothScroller from "./path/to/SmoothScroller";
 *
 * const { start } = setup({ override: { scroller: SmoothScroller } });
 * const { scroller } = await start();
 * console.log( scroller instanceof SmoothScroller );
 * >> true
 * ```
 *
 * 上記のように start 関数によって初期化が完了した後は以下の方法で SmoothScroller のインスタンスを取得することができます。
 * After initialization is complete using the start function as shown above, you can obtain an instance of SmoothScroller using the following method.
 *
 * **取得方法(How to get)**
 * ```ts
 * import { scroller } from "negl";
 * // or
 * window.negl.scroller
 * ```
 */
class SmoothScroller extends Scroller {
  scrollBar: SmoothScrollbar;

  constructor() {
    super();

    /**
     * smooth-scrollbar をマウントする対象の要素を取得
     * 注意）.page-container はページトランジション時に削除されるため、.page-containerの親要素に scrollbar をマウントする必要あり。
     *
     * Get the element to mount smooth-scrollbar on
     * Note: Since .page-container is removed during page transitions, the scrollbar needs to be mounted on the parent element of .page-container.
     */
    const scrollContainerSelector = "#scroll-container";
    const scrollContainer = document.querySelector(
      scrollContainerSelector
    ) as HTMLElement;
    if (!scrollContainer) {
      throw new Error(
        `スクロールのマウント先要素[${scrollContainerSelector}]が見つかりません。`
      );
    }

    this.initScrollbarPlugin();

    this.updateStyle(scrollContainer);

    this.scrollBar = this.mountScrollBar(scrollContainer);

    this.bindEvents();
  }

  bindEvents() {
    this.scrollBar.addListener(this.onScroll);

    hook.on(
      hook.T_BOTH_DOM_EXIST,
      () => {
        /**
         * ページトランジション時には .page-container が遷移前後で入れ替わるため、スクロールバーの情報も更新する
         * During page transitions, since .page-container is swapped before and after the transition, the scrollbar information is also updated.
         * [scrollbar.update](https://github.com/idiotWu/smooth-scrollbar/blob/66c67b85d35c14486bd25a73806c0ab13ffeb267/docs/api.md#scrollbarupdate)
         */
        this.scrollBar.update();
      },
      { priority: 9999 }
    );
  }

  /**
   * プラグインの初期化
   * Initialization of the plugin
   */
  initScrollbarPlugin() {
    const _self = this;
    class DisablePlugin extends SmoothScrollbar.ScrollbarPlugin {
      static pluginName = "disable";

      static defaultOptions = {
        disable: false,
      };

      transformDelta(delta: { x: number; y: number }) {
        return this.options.disable ? { x: 0, y: 0 } : delta;
      }
      onRender(_remainMomentum: Data2d): void {
        /**
         * スクロールの変化量を更新
         * Update the amount of scroll change
         */
        _self.setDelta(_remainMomentum);
      }
    }

    SmoothScrollbar.use(DisablePlugin);
  }

  /**
   * smooth-scrollbar 用のスタイル
   * Styles for smooth-scrollbar
   * @param scrollContainer
   */
  updateStyle(scrollContainer: HTMLElement) {
    const globalContainer = INode.qs(config.$.globalContainer) as HTMLElement;
    document.body.style.overflow = "hidden";
    globalContainer.style.overflow = "hidden";
    scrollContainer.style.height = "100vh";
  }

  /**
   * smooth-scrollbar をマウント
   * Mount smooth-scrollbar
   * @param scrollContainer
   * @returns
   */
  mountScrollBar(scrollContainer: HTMLElement) {
    return SmoothScrollbar.init(
      scrollContainer,

      { delegateTo: document }
    );
  }

  /**
   * スクロールを無効化します。（スクロールできないようになります。）
   * Disables scrolling. (Makes it impossible to scroll.)
   */
  disable() {
    this.scrollBar.updatePluginOptions("disable", {
      disable: true,
    });
  }
  /**
   * スクロールを有効化します。（スクロールできるようになります。）
   * Enables scrolling. (Makes it possible to scroll.)
   */
  enable() {
    this.scrollBar.updatePluginOptions("disable", {
      disable: false,
    });
  }

  /**
   * [scrollIntoView](https://developer.mozilla.org/ja/docs/Web/API/Element/scrollIntoView) メソッドと同等の機能を持ちます。
   * Has the same functionality as the [scrollIntoView](https://developer.mozilla.org/en-US/docs/Web/API/Element/scrollIntoView) method.
   * scrollIntoView() が呼び出された要素がユーザーに見えるところまで、要素の親コンテナーをスクロールします。
   * Scrolls the parent container of the element until the element that called scrollIntoView() is visible to the user.
   * @param el
   * @param scrollIntoViewOptions [scrollIntoViewOptions](https://developer.mozilla.org/ja/docs/Web/API/Element/scrollIntoView)
   * ただし、behavior オプションは "auto" のみ許容します。
   * However, only the "auto" option is allowed for the behavior option.
   */
  async scrollIntoView(
    el: HTMLElement,
    scrollIntoViewOptions?: Record<string, unknown>
  ) {
    this.scrollBar.scrollIntoView(el, scrollIntoViewOptions);
    // スクロール位置が変更されたたため、hook.trigger を発火
    // Fires hook.trigger because the scroll position has changed
    const originalDamping = config.scroller.damping;
    config.scroller.damping = 1;
    await hook.trigger(hook.SCROLL, this);
    config.scroller.damping = originalDamping;
  }

  /**
   * ページの最上部から px 分スクロールを行います。
   * Scrolls down from the top of the page by px.
   * @param px 画面最上部からスクロールしたい量を px で指定します。(Specify the amount you want to scroll from the top of the screen in px.)
   */
  set scrollTop(px: number) {
    if (typeof px !== "number")
      throw new Error(`this.scrollTopに数値以外が設定されています。値: ${px}`);

    this.scrollBar.scrollTop = px;
    // スクロール位置が変更されたたため、hook.trigger を発火
    // Fires hook.trigger because the scroll position has changed
    void (async () => {
      const originalDamping = config.scroller.damping;
      config.scroller.damping = 1;
      await hook.trigger(hook.SCROLL, this);
      config.scroller.damping = originalDamping;
    })();
  }

  /**
   * ページの最上部からのスクロール量を数値で取得します。
   * Gets the scroll amount from the top of the page as a number.
   * @returns ページの最上部からのスクロール量(The scroll amount from the top of the page)
   */
  get scrollTop() {
    return this.scrollBar.scrollTop;
  }
}

export { SmoothScroller };
