/**
 * スクロールアニメーションの制御ファイル
 * 概要：data-scroll-trigger属性によって実行される処理
 *
 * Scroll animation control file
 * Overview: Processes executed by the data-scroll-trigger attribute
 */
import gsap from "gsap";
import { ScrollTrigger } from "gsap/ScrollTrigger";
import { world } from "negl";
import { ScrollTriggerFactory } from "#/parts/helper/scroll/scroll-trigger";
import { isMobile } from "#/parts/helper/isMobile";
import initRipplePass from "#/parts/glsl/ripple";

let startTrigger: string = isMobile() ? "top 80%" : "top 70%";

const progress: ScrollTriggerFactory = (target) => {
  function onEnter() {
    const o = world.getObByEl(target);
    if (!o) return;
    gsap.to(o.uniforms.uProgress, {
      value: 1,
    });
  }

  function onLeaveBack() {
    const o = world.getObByEl(target);
    if (!o) return;
    gsap.to(o.uniforms.uProgress, {
      value: 0,
    });
  }

  const st = ScrollTrigger.create({
    trigger: target,
    start: startTrigger,
    onEnter,
    onLeaveBack,
    onKill: onLeaveBack,
  });

  return st;
};

const progressSplit: ScrollTriggerFactory = (target) => {
  function onEnter() {
    const o = world.getObByEl(target);
    if (!o) return;
    gsap.to(o.uniforms.uProgress, {
      value: 1,
    });
  }
  function onLeaveBack() {
    const o = world.getObByEl(target);
    if (!o) return;
    gsap.to(o.uniforms.uProgress, {
      value: 0,
    });
  }
  const st = ScrollTrigger.create({
    trigger: target,
    start: "center center",
    onEnter,
    onLeaveBack,
    onKill: onLeaveBack,
  });
  return st;
};

const fade: ScrollTriggerFactory = (target) => {
  function onEnter() {
    target.classList.add("inview");
  }

  function onLeaveBack() {
    target.classList.remove("inview");
  }

  const st = ScrollTrigger.create({
    trigger: target,
    start: startTrigger,
    onEnter,
    onLeaveBack,
    onKill: onLeaveBack,
  });

  return st;
};

const progressParticles: ScrollTriggerFactory = (target) => {
  function onEnter() {
    const o = world.getObByEl(target);
    if (!o) return;
    o.goTo(1, 1);
  }

  function onEnterBack() {
    const o = world.getObByEl(target);
    if (!o) return;
    o.goTo(0, 1);
  }

  const st = ScrollTrigger.create({
    trigger: target,
    start: "center center",
    end: "center center",
    onEnter,
    onEnterBack,
    onKill: onEnterBack,
  });
  return st;
};

const playVideo: ScrollTriggerFactory = (target) => {
  const o = world.getObByEl(target);
  if (!o) return;

  const video = o.uniforms.tex1.value.source.data as HTMLVideoElement;

  function play() {
    video.paused && video?.play();
  }

  function pause() {
    video?.pause();
  }

  const st = ScrollTrigger.create({
    trigger: target,
    start: startTrigger,
    onEnter: play,
    onEnterBack: play,
    onLeave: pause,
    onLeaveBack: pause,
    onKill: pause,
  });

  return st;
};

const logo: ScrollTriggerFactory = (target) => {
  function onEnter() {
    target.classList.add("inview");
  }

  function onEnterBack() {
    target.classList.remove("inview");
  }

  const st = ScrollTrigger.create({
    trigger: target,
    start: "center top",
    onEnter,
    onEnterBack,
    onKill: onEnterBack,
  });
  return st;
};

const reversal: ScrollTriggerFactory = (target) => {
  const fresnel = world.getObByEl(".fresnel");
  const skillTitle = world.getObByEl(".skill__title-text");
  const graphicTitle = world.getObByEl(".graphic__title-text");

  const reversal = { value: 0 };
  function onUpdate() {
    fresnel && (fresnel.uniforms.uReversal.value = reversal.value);
    skillTitle && (skillTitle.uniforms.uReversal.value = reversal.value);
    graphicTitle && (graphicTitle.uniforms.uReversal.value = reversal.value);
  }

  function onEnter() {
    gsap.to(":root", {
      "--c-text": "#333",
      "--c-sec": "rgba(51,51,51,0.8)",
      "--c-main": "black",
      "--c-bg": "radial-gradient(#fff, #e5e5e5)",
    });
    gsap.to(reversal, {
      value: 1,
      onUpdate,
    });
  }

  function onLeaveBack() {
    gsap.to(":root", {
      "--c-text": "#dadada",
      "--c-sec": "rgba(218,218,218,0.8)",
      "--c-main": "white",
      "--c-bg": "radial-gradient(#000, #191919)",
    });
    gsap.to(reversal, {
      value: 0,
      onUpdate,
    });
  }

  const st = ScrollTrigger.create({
    trigger: target,
    start: startTrigger,
    onEnter,
    onLeaveBack,
  });

  onLeaveBack();

  return st;
};

const ripple: ScrollTriggerFactory = async (target) => {
  if (isMobile()) return;

  const { addPass: onEnterBack, removePass: onEnter } = await initRipplePass();

  const st = ScrollTrigger.create({
    trigger: target,
    start: "center center",
    end: "center center",
    onEnter,
    onEnterBack,
    onKill: onEnter,
  });

  return st;
};

export {
  progress,
  fade,
  progressParticles,
  playVideo,
  logo,
  reversal,
  ripple,
  // progressSplit,
};
