import { INode, utils, Scroller, hook } from "negl";
import gsap from "gsap";
import ScrollTrigger from "gsap/ScrollTrigger";
import { SmoothScroller } from "./SmoothScroller";
import gsapScrollTrigger from "gsap/ScrollTrigger";

export type onDestroy = () => void | Promise<void>;

/**
 * target に対して ScrollTrigger アニメーションを設定する関数の型
 * Type definition for a function that sets up ScrollTrigger animations for a target
 */
export type ScrollTriggerFactory = (
  target: HTMLElement
) => ScrollTrigger | undefined | Promise<ScrollTrigger | undefined>;

export type ScrollTriggerFactories = { [type: string]: ScrollTriggerFactory };

/**
 * ScrollTrigger の初期化を行います。
 * Initializes ScrollTrigger.
 * @param scroller
 */
export function init(scroller: Scroller) {
  gsap.registerPlugin(ScrollTrigger);
  syncScrollBar(scroller);
  bindEvents(scroller);
}

/**
 * smooth-scrollbar に ScrollTrigger を紐づけるために必要な処理を実行します。
 * Executes the necessary steps to link ScrollTrigger with smooth-scrollbar.
 * @param scroller Scroller instance
 */
function syncScrollBar(scroller: Scroller) {
  if (!(scroller instanceof SmoothScroller)) return;

  const scrollContainer = scroller.scrollBar.containerEl;
  ScrollTrigger.scrollerProxy(scrollContainer, {
    scrollTop(value) {
      if (arguments.length) {
        scroller.scrollTop = value!;
      }
      return scroller.scrollTop;
    },
    getBoundingClientRect() {
      return {
        top: 0,
        left: 0,
        width: window.innerWidth,
        height: window.innerHeight,
      };
    },
  });

  ScrollTrigger.defaults({
    scroller: scrollContainer,
  });
}

let scrollTriggerFactories: ScrollTriggerFactories = {};
/**
 * ScrollTriggerアニメーションのパターンを追加します。
 * Adds patterns for ScrollTrigger animations.
 * @param factories
 */
export function addActions(factories: ScrollTriggerFactories) {
  Object.assign(scrollTriggerFactories, factories);
}

/**
 * ScrollTriggerアニメーションを要素にマウントします。
 * Mounts ScrollTrigger animations to elements.
 * @param scope
 */
export function mountAnimations(scope?: HTMLElement) {
  ScrollTrigger.getAll().forEach((st) => {
    st.enable(false, false);
  });
  /**
   * data-scroll-trigger属性が付与されている要素を全て取得
   * Retrieves all elements with the data-scroll-trigger attribute.
   */
  const els = INode.qsAll(`[data-scroll-trigger]`, scope);
  els.forEach((el) => {
    const key = utils.toCamelCase("scroll-trigger");
    const types = INode.getDS(el, key)!.split(",");
    types.forEach(async (type) => {
      // type アニメーション
      // type animation
      const scrollTriggerFactory = scrollTriggerFactories?.[type];
      const st = await scrollTriggerFactory?.(el as HTMLElement);
      st?.update?.();
    });
  });
}

const destroyStTargets: ScrollTrigger[] = [];
/**
 * ページ遷移時のScrollTriggerインスタンスの無効化を行います。
 * Disables ScrollTrigger instances during page transitions.
 * @param currentPageContainer 前のページの .page-container 要素. (The .page-container element of the previous page.)
 */
function disable(currentPageContainer: HTMLElement) {
  // スクロールトリガーの状態を無効化
  // Disables the state of the scroll trigger
  ScrollTrigger.getAll().forEach((st) => {
    /**
     * ページトランジション中に前のページの .page-container が削除されるため、
     * 削除される前ににスクロールトリガーのアニメーションを切って不要なアニメーションが行われないようにする。
     *
     * During page transitions, the previous page's .page-container is removed,
     * so disable the scroll trigger animations before it is removed to prevent unnecessary animations.
     */
    st.disable();

    // ページコンテナ内の ScrollTrigger の場合
    // In the case of ScrollTrigger inside the page container
    const inPageContainer = currentPageContainer!.contains(
      st.trigger as HTMLElement
    );
    if (inPageContainer) {
      // 後に削除する ScrollTrigger のインスタンスを追加
      // Add instances of ScrollTrigger to be deleted later
      destroyStTargets.push(st);
    }
  });
}

/**
 * ページ遷移時のScrollTriggerインスタンスの削除を行います。
 * Deletes ScrollTrigger instances during page transitions.
 */
async function destroy() {
  let st;

  // 前のページの ScrollTrigger を削除
  // Delete the previous page's ScrollTrigger
  while ((st = destroyStTargets.shift())) {
    st.kill(true);
  }
}

function bindEvents(scroller: Scroller) {
  if (scroller instanceof SmoothScroller) {
    // SmoothScroller と併用する際は ScrollTrigger のスクロール情報を強制的に更新
    // Force update of ScrollTrigger's scroll information when used with SmoothScroller
    hook.on(hook.SCROLL, ScrollTrigger.update);
  }

  // ScrollTrigger.pin が画面内に存在すると pin を生成してから、スクロール位置を再調整する必要あり。
  // When ScrollTrigger.pin exists on the screen, it is necessary to create the pin and then readjust the scroll position.
  hook.on(
    hook.T_BOTH_OB_EXIST,
    () => {
      // スクロール位置調整のため、一度だけスクロール処理を実行
      // Execute the scroll process once for scroll position adjustment
      hook.trigger(hook.SCROLL, scroller);
      // transition-base.js の revealNext（次のページの表表示）より先に実行する必要があるため、priority を 100 に設定
      // Set priority to 100 because it needs to be executed before revealNext (display of the next page) in transition-base.js
    },
    { priority: 100 }
  );

  // 現在ページのScrollTriggerイベントの状態を変更しないようにする
  // Prevent changes to the current page's ScrollTrigger event state
  hook.on(hook.T_BEGIN, ({ current: { el } }) => {
    if (!el) return;
    return disable(el);
  });

  // 新しいページのスクロールアニメーションを追加
  // Add scroll animations for the new page
  hook.on(
    hook.T_BOTH_OB_EXIST,
    ({ next: { el } }) => {
      if (!el) return;
      mountAnimations(el);
    },
    { priority: 200 }
  );

  // 前のページのScrollTriggerイベントを削除
  // Remove ScrollTrigger events from the previous page
  hook.on(hook.T_END, async () => await destroy());
}
