/**
 * メニュー制御ファイル
 * 概要：メニューの開閉処理
 *
 * Menu control file
 * Overview: Menu open/close processing
 */
import gsap from "gsap";
import { INode, config, World, Scroller } from "negl";

export interface Menu {
  isOpen: boolean;
  init(args: MenuInitArgs): void;
  toggle(): Promise<void>;
  reset(): void;
}

type MenuInitArgs = { world?: World; scroller?: Scroller };

let world: World | null = null,
  scroller: Scroller | null = null,
  clickTl: gsap.core.Timeline | null = null,
  disable = false;

const menu: Menu = {
  init,
  toggle,
  reset,
  isOpen: false,
};

const $: { [key: string]: HTMLElement | HTMLElement[] | null } = {};

function init({ world: _world, scroller: _scroller }: MenuInitArgs) {
  world = _world || null;
  scroller = _scroller || null;
  $.container = INode.getElement(config.$.globalContainer) as HTMLElement;
  $.btn = INode.getElement(".btn-menu") as HTMLElement;
  $.wraps = INode.qsAll(".btn-menu__wrap") as HTMLElement[];
  $.bars = INode.qsAll(".btn-menu__bar") as HTMLElement[];

  clickTl = _createClickTL();
  _bindEvents();
}

function _bindEvents() {
  if ($.btn) {
    ($.btn as HTMLButtonElement).addEventListener(config.event.click, () =>
      toggle()
    );
    ($.btn as HTMLButtonElement).addEventListener("pointerenter", () =>
      _enter()
    );
  }
}

function _toggleMeshVisibility(isOpen: boolean) {
  if (world) {
    const fvText = world.getObByEl(".fv__text-shader");
    const titleEls = INode.qsAll(`[data-${config.prefix.ob}="distortion"]`);
    titleEls.forEach((titleEl) => {
      const titleObj = world!.getObByEl(titleEl as HTMLElement);
      if (titleObj) {
        titleObj.mesh.visible = isOpen;
      }
    });
    if (fvText) {
      fvText.mesh.visible = isOpen;
    }
  }
}

function _createClickTL() {
  const tl = gsap.timeline({
    paused: true,
    defaults: {
      duration: 0.3,
    },
  });

  tl.to(
    ($.wraps as HTMLElement[])[0],
    {
      y: 0,
      rotateZ: 225,
    },
    "toggle"
  )
    .to(
      ($.wraps as HTMLElement[])[1],
      {
        x: "-1em",
        opacity: 0,
      },
      "toggle"
    )
    .to(
      ($.wraps as HTMLElement[])[2],
      {
        y: 0,
        rotateZ: -45,
      },
      "toggle"
    )
    .to(config.$.pageContainer, {
      opacity: 0,
      duration: 0.1,
    });

  return tl;
}

async function toggle(): Promise<void> {
  return new Promise((resolve) => {
    if ($.container) {
      ($.container as HTMLElement).classList.toggle("menu-open");
    }

    if (menu.isOpen) {
      setTimeout(() => {
        _toggleMeshVisibility(true);
        if (clickTl) {
          clickTl.vars.onReverseComplete = resolve;
          clickTl.reverse();
        }
        if (scroller) {
          scroller.enable();
        }
      }, 1000);
    } else {
      setTimeout(() => {
        _toggleMeshVisibility(false);
        if (scroller) {
          scroller.disable();
        }
      }, 1000);
      if (clickTl) {
        clickTl.vars.onComplete = resolve;
        clickTl.play();
      }
    }
    menu.isOpen = !menu.isOpen;
  });
}

function _enter() {
  const tl = gsap.timeline({
    defaults: {
      stagger: 0.1,
      duration: 0.3,
    },
  });
  tl.set($.bars, {
    transformOrigin: "right",
  })
    .to($.bars, {
      scaleX: 0,
    })
    .set($.bars, {
      transformOrigin: "left",
    })
    .to($.bars, {
      scaleX: 1,
    });
}

function reset() {
  if (disable) return;
  clickTl = _createClickTL();
}

export { menu };
