﻿/**
 * 2重の円が表示されるマウスエフェクト
 * A mouse effect where double circles are displayed
 *
 * data-mouse="highlight" data-mouse
 *
 *
 * アニメーションを付与することもできます。
 * You can also add animations.
 *
 * ## エフェクトのタイプ(Types of Effects)
 * - highlight: 円が大きくなり色が変わります。(The circle grows larger and changes color.)
 * - stuck: 円が要素の大きさに合わせて止まります。(The circle stops according to the size of the element.)
 * - hide: 円が非表示になります。(The circle becomes invisible.)
 *
 * ```html
 * <div data-mouse="stuck"></div> // div の大きさで円が止まります。(The circle stops according to the size of the div.)
 * <div data-mouse="highlight"></div> // 円が大きくなります。(The circle grows larger.)
 * <div data-mouse="highlight" data-mouse-scale="3"></div> // 円が3倍に大きくなります。(The circle grows three times larger.)
 * ```
 */
import { INode, utils } from "negl";

let $ = {},
  initial = {},
  shouldTrackMousePos = true;
const distortion = { level: 500, max: 0.4 };

function initMouseCircle(negl, hideDefaultCursor = false) {
  const { mouse, config, utils, viewport } = negl;
  if (utils.isTouchDevices()) {
    return () => {};
  }

  const { current, target, delta } = mouse;

  initial = {
    x: viewport.width / 2,
    y: viewport.height / 2,
    r: 40,
    fill: "#ffffff",
    fillOpacity: 0,
    strokeWidth: 1,
    scale: 1,
    mixBlendMode: "difference",
  };

  Object.assign(current, initial);
  Object.assign(target, initial);
  Object.assign(delta, { scale: 1, fillOpacity: 0 });

  const svg = _createHTML(negl);
  svg.style.mixBlendMode = initial.mixBlendMode;
  const circles = INode.qsAll("circle", svg);
  const outerCircle = circles[0];
  const innerCircle = circles[1];
  const globalContainer = INode.getElement(config.$.globalContainer);

  globalContainer.append(svg);

  if (hideDefaultCursor) {
    document.body.style.cursor = "none";
  }

  const transforms = INode.qsAll(`[data-mouse]`);

  globalContainer.append(svg);

  $ = {
    svg,
    outerCircle,
    innerCircle,
    globalContainer,
    transforms,
  };

  bindEvents(negl);

  /**
   * ロード完了時に実行
   * Executed upon load completion
   * @description ユーザー定義メソッドのため実行必須ではありません。(Not required to execute because it is a user-defined method.)
   */
  return function onload() {
    if (utils.isTouchDevices()) return;
    const intervalId = window.setInterval(() => {
      if (!mouse.isUpdate()) return;
      $.svg.style.opacity = 1;
      clearInterval(intervalId);
    }, 300);
  };
}

function bindEvents(negl) {
  const { mouse, hook } = negl;
  /**
   * マウス位置への追従処理
   * Mouse position tracking process
   */
  function bindMove() {
    move(mouse);
  }
  hook.on(hook.RENDER, bindMove);

  /**
   * 画面リサイズ時の処理
   * Processing during screen resizing
   */
  hook.on(hook.RESIZE, () => _resize(negl));

  /**
   * ページ遷移時の処理
   * Processing during page transitions
   */
  let targets;
  const removeEventFns = [];
  const handlers = getMouseAnimationHandlers();
  /**
   * HTML要素に入った際のアニメーションの設定
   * Setting animations when entering HTML elements
   */
  function _bindDOMEvents() {
    const { mouse } = negl;

    targets = [...document.querySelectorAll("[data-mouse], a, button")];
    targets.forEach((el) => {
      const handlerType = INode.getDS(el, "mouse") ?? "highlight";
      const handler = handlers[handlerType];

      if (!handler) return;

      Object.entries(handler).forEach(([mouseType, action]) => {
        const fn = (event) => action(mouse, event);
        el.addEventListener(`pointer${mouseType}`, fn);
        // イベントを削除するための関数を準備（ページ遷移の際に使用）
        // Prepare a function to remove the event (used during page transitions)
        removeEventFns.push(() => {
          el.removeEventListener(`pointer${mouseType}`, fn);
        });
      });
    });
  }

  /**
   * ページ遷移時のイベント削除、スタイルの初期化関数
   * Event removal and style initialization function during page transitions
   */
  function _destroy() {
    let removeEvent;
    // DOMにバインドしたイベントを削除
    // Remove events bound to the DOM
    while ((removeEvent = removeEventFns.shift())) {
      removeEvent();
    }

    // マウスのスタイルを初期化
    // Initialize the mouse style
    handlers.stuck.leave(mouse);
    // マウスを透明に変更
    // Change the mouse to transparent
    $.svg.style.opacity = 0;
  }

  _bindDOMEvents();
  // ページ遷移が開始された時
  // When page transition starts
  hook.on(hook.T_BEGIN, _destroy);
  // ページ遷移が終了する時
  // When page transition ends
  hook.on(hook.T_END, _bindDOMEvents);
  hook.on(hook.T_END, () => {
    $.svg.style.opacity = 1;
  });
}

/**
 * svgの作成(Creation of SVG)
 * @description ユーザー独自メソッド(User-defined method)
 */
function _createHTML({ mouse, viewport }) {
  const { current } = mouse;

  return INode.htmlToEl(`
    <svg
      class="mouse-viewport"
      width="${viewport.width}"
      height="${viewport.height}"
      preserveAspectRatio="none meet"
      viewBox="0 0 ${viewport.width} ${viewport.height}"
      style="opacity: 0; transition: opacity 1s;"
      >
      <g class="mouse-wrapper">
        <circle
          class="circle outer"
          r="${current.r}"
          cx="${current.x}"
          cy="${current.y}"
          fill="${current.fill}"
          fill-opacity="${current.fillOpacity}"
          stroke="${current.fill}"
          stroke-width="${current.strokeWidth}"
          style="transform-origin: ${current.x}px ${current.y}px"
        ></circle>
        <circle
          class="circle outer"
          r="${3}"
          cx="${current.x}"
          cy="${current.y}"
          fill="${current.fill}"
          style="transform-origin: ${current.x}px ${current.y}px"
        ></circle>
      </g>
    </svg>
  `);
}

/**
 * マウスが動いた際に実行
 * Executed when the mouse moves
 */
function move(mouse) {
  if (utils.isTouchDevices()) return;
  const { current, target, delta, speed } = mouse;
  // 値の更新(Update values)
  // delta.x = target.x - current.x; // delta.xの更新はnegl内部で行われるため不要(Not required because the update of delta.x is done internally in negl.)
  // delta.y = target.y - current.y; // delta.yの更新はnegl内部で行われるため不要(Not required because the update of delta.y is done internally in negl.)
  delta.scale = target.scale - current.scale;
  delta.fillOpacity = target.fillOpacity - current.fillOpacity;

  // current.x += delta.x * speed; // delta.xの更新はnegl内部で行われるため不要(Not required because the update of delta.x is done internally in negl.)
  // current.y += delta.y * speed; // delta.yの更新はnegl内部で行われるため不要(Not required because the update of delta.y is done internally in negl.)
  current.scale += delta.scale * speed;
  current.fillOpacity += delta.fillOpacity * speed;

  let distort =
    Math.sqrt(Math.pow(delta.x, 2) + Math.pow(delta.y, 2)) / distortion.level;
  distort = Math.min(distort, distortion.max);
  current.scaleX = (1 + distort) * current.scale;
  current.scaleY = (1 - distort) * current.scale;

  current.rotate = (Math.atan2(delta.y, delta.x) / Math.PI) * 180;

  // スタイルの変更
  // Style changes
  $.innerCircle.setAttribute("cx", target.x);
  $.innerCircle.setAttribute("cy", target.y);

  $.outerCircle.style.transformOrigin = `${current.x}px ${current.y}px`;
  $.outerCircle.setAttribute("cx", current.x);
  $.outerCircle.setAttribute("cy", current.y);
  $.outerCircle.setAttribute("fill-opacity", current.fillOpacity);

  const rotate = `rotate(${current.rotate}deg)`;
  const scale = `scale(${current.scaleX}, ${current.scaleY})`;

  $.outerCircle.style.transform = `${rotate} ${scale}`;
}

/**
 * 画面のリサイズ時に行う処理
 * Processing to be done when resizing the screen
 */
function _resize({ viewport, utils }) {
  if (utils.isTouchDevices()) {
    return;
  }
  $.svg.setAttribute("width", viewport.width.toString());
  $.svg.setAttribute("height", viewport.height.toString());
  $.svg.setAttribute("viewBox", `0 0 ${viewport.width} ${viewport.height}`);
}

/**
 * マウスカーソルのアニメーション制御
 * 概要：data-mouse属性によって実行される処理
 *
 * Mouse cursor animation control
 * Overview: Processes executed by the data-mouse attribute
 */
function getMouseAnimationHandlers() {
  const highlight = {
    enter: (mouse, { currentTarget }) => {
      const scale = INode.getDS(currentTarget, `mouseScale`) || 1;

      $.innerCircle.style.visibility = "hidden";
      mouse.setTarget({
        scale: scale,
        fillOpacity: 1,
      });
    },
    leave: (mouse) => {
      $.innerCircle.style.visibility = "visible";

      mouse.setTarget({
        scale: initial.scale,
        fillOpacity: initial.fillOpacity,
      });
    },
  };

  const stuck = {
    enter: (mouse, { currentTarget }) => {
      mouse.stopTrackMousePos();

      const scale = INode.getDS(currentTarget, `mouseScale`) || 1;

      const rect = INode.getRect(currentTarget);

      $.innerCircle.style.visibility = "hidden";
      mouse.setTarget({
        x: rect.x + rect.width / 2,
        y: rect.y + rect.height / 2,
        scale: (rect.width / 2 / initial.r) * scale,
        fillOpacity: 1,
      });
    },
    leave: (mouse) => {
      $.innerCircle.style.visibility = "visible";

      mouse.setTarget({
        scale: initial.scale,
        fillOpacity: initial.fillOpacity,
      });
      mouse.startTrackMousePos();
    },
  };

  const hide = {
    enter: (mouse) => {
      $.innerCircle.style.visibility = "hidden";

      mouse.setTarget({
        scale: 0,
      });
    },
    leave: (mouse) => {
      $.innerCircle.style.visibility = "visible";

      mouse.setTarget({
        scale: initial.scale,
      });
    },
  };

  const handlers = {
    highlight,
    stuck,
    hide,
  };

  return handlers;
}

export default initMouseCircle;
