/**
 * スライダ制御ファイル
 * 概要：スライダの次へボタン、スクロール量によるスライダの遷移制御
 * ※余り汎用的ではないので、スライダの制御に合わせて関数を作成してください。
 *
 * File for controlling the slider
 * Overview: Controls the slider transitions through the next button and scroll amount.
 * ※Since it's not very generic, please create functions according to the slider control.
 */
import gsap from "gsap";
import ScrollTrigger from "gsap/ScrollTrigger";

import { world, INode, config, hook, meshUI } from "negl";

interface NavHandlerOptions {
  sliderSelector: string;
  prevBtnSelector: string;
  nextBtnSelector: string;
  textSelector: string;
}

function mountNavBtnHandler({
  sliderSelector,
  prevBtnSelector,
  nextBtnSelector,
  textSelector,
}: NavHandlerOptions) {
  const prevEl = INode.getElement(prevBtnSelector);
  const nextEl = INode.getElement(nextBtnSelector);

  const slider = world.getObByEl(sliderSelector)!;
  const text = world.getObByEl(textSelector);

  function goTo(idx: number) {
    slider?.goTo?.(idx);
    text?.goTo?.(idx);
  }

  prevEl && prevEl.addEventListener(config.event.click, goPrevByNavBtn);
  nextEl && nextEl.addEventListener(config.event.click, goNextByNavBtn);

  function goPrevByNavBtn() {
    const idx = slider.activeSlideIdx - 1;
    goTo(idx);
    // メッシュに反応しないようにイベント伝播を停止
    // Stop event propagation to prevent interaction with the mesh
    meshUI.stopPropagation();
  }

  function goNextByNavBtn() {
    const idx = slider.activeSlideIdx + 1;
    goTo(idx);
    // メッシュに反応しないようにイベント伝播を停止
    // Stop event propagation to prevent interaction with the mesh
    meshUI.stopPropagation();
  }

  hook.on(hook.T_BEGIN, destroy, { once: true });

  function destroy() {
    prevEl!.removeEventListener(config.event.click, goPrevByNavBtn);
    nextEl!.removeEventListener(config.event.click, goNextByNavBtn);
  }

  return { goTo };
}

function mountSkillBtnHandler(
  sliderSelector: string,
  prevBtnSelector: string,
  nextBtnSelector: string,
  textSelector: string
) {
  const prevEl = INode.getElement(prevBtnSelector) as HTMLElement;
  const nextEl = INode.getElement(nextBtnSelector) as HTMLElement;

  const slider = world.getObByEl(sliderSelector)!;
  const slideUl = INode.getElement(textSelector) as HTMLElement;
  const slideLis = Array.from(slideUl.children) as HTMLElement[];

  let translateX = 50;
  let prevIdx = 0;
  slideLis.forEach((li, i) => {
    li.style.transform = `translateX(-${i * translateX}px)`;
  });
  function goTo(idx: number) {
    slider.goTo(idx);
  }

  prevEl.addEventListener("click", () => {
    let idx = slider.activeSlideIdx - 1;
    idx = (slider.texes.size + idx) % slider.texes.size;
    goTo(idx);
    slideLis[idx].style.opacity = "1";
    slideLis[prevIdx].style.opacity = "0";
    slideUl.style.transform = `translateX(${idx * translateX}px)`;
    prevIdx = idx;
  });

  nextEl.addEventListener("click", () => {
    let idx = slider.activeSlideIdx + 1;
    idx = idx % slider.texes.size;
    goTo(idx);
    slideLis[idx].style.opacity = "1";
    slideLis[prevIdx].style.opacity = "0";
    slideUl.style.transform = `translateX(${idx * translateX}px)`;
    prevIdx = idx;
  });
}

function mountScrollHandler({
  sliderSelector,
  triggerSelector,
  textSelector,
}: {
  sliderSelector: string;
  triggerSelector: string;
  textSelector: string;
}) {
  const slider = world.getObByEl(sliderSelector)!;
  const slideUl = INode.getElement(textSelector) as HTMLElement;
  const slideLis = Array.from(slideUl.children) as HTMLElement[];

  let translateX = 50;
  let prevIdx = 0;
  slideLis.forEach((li, i) => {
    li.style.transform = `translateX(-${i * translateX}px)`;
    li.style.opacity = i === 0 ? "1" : "0";
  });
  slideUl.style.transform = "translateX(0px)";

  function goTo(idx: number) {
    slider.goTo(idx);
  }

  goTo(0);

  const slides = { idx: 0 };
  const animation = gsap.to(slides, {
    idx: slideLis.length - 1,
  });
  const st = ScrollTrigger.create({
    animation,
    trigger: triggerSelector,
    start: "top 0%",
    end: "+=3000",
    pin: true,
    scrub: true,
    onUpdate: () => {
      let idx = Math.round(slides.idx);
      idx = (slider.texes.size + idx) % slider.texes.size;
      if (idx === prevIdx) return;
      goTo(idx);
      slideLis[idx].style.opacity = "1";
      slideLis[prevIdx].style.opacity = "0";
      slideUl.style.transform = `translateX(${idx * translateX}px)`;
      prevIdx = idx;
    },
  });
}

export { mountNavBtnHandler, mountSkillBtnHandler, mountScrollHandler };
