﻿import gsap from "gsap";
import initDistortionPass from "#/parts/glsl/distortion/pass";

function registTransitionDistortion({ hook, INode, config, world }) {
  const type = "distortion";
  // トランジション 初期時（1回のみ実行）
  // Transition initialization (executed only once)
  hook.on(hook.T_INIT, initPass);

  // トランジション 開始時
  // At the start of the transition
  hook.on(hook.T_BEGIN, keepOb, { type, priority: 100 });
  hook.on(hook.T_BEGIN, fadeOutByDistortionPass, { type });

  // トランジション 次のページのOBインスタンス挿入後
  // After inserting the OB instance of the next page during the transition
  hook.on(hook.T_BOTH_OB_EXIST, revealNext, { type });
  hook.on(hook.T_BOTH_OB_EXIST, enterLeave, { type });

  // パスの初期化
  // Initialization of the pass
  let _distortion;
  async function initPass() {
    _distortion = await initDistortionPass({ world });
    _distortion.setProgress(1);
    _distortion.removePass(false);
  }

  /**
   * 背景のFresnelの動画を残したままにします。
   * Keeps the background Fresnel video intact.
   * @param [TransitionState](https://docs.not-equal.dev/reference/interfaces/TransitionState/)
   */
  function keepOb({ current }) {
    // 次のページでも保持し続けるObのセレクター
    // Selector for the Ob that will continue to be retained on the next page
    const keepObs = [".fresnel"];
    keepObs.forEach((selector) => {
      const ob = world.getObByEl(selector);
      const idx = current.os.indexOf(ob);
      // current.os に含まれるエフェクトは T_BOTH_OB_EXIST フックの完了後、自動的にシーンから除去されるため、current.os からシーン内に残したいエフェクトを削除します。
      // Effects included in current.os are automatically removed from the scene after the T_BOTH_OB_EXIST hook is completed, so remove the effects you want to keep in the scene from current.os.
      // https://docs.not-equal.dev/tutorial/transition/#t_both_ob_exist次のページの-ob-オブジェクトがアクセス可能な状態
      current.os.splice(idx, 1);
    });
  }

  // ディストーションのトランジション
  // Distortion transition
  async function fadeOutByDistortionPass({ current }) {
    _distortion.addPass();
    const progress = { value: 1 };
    return new Promise((resolve) => {
      gsap.to(progress, {
        value: 0,
        duration: 0.5,
        ease: "power3.in",
        onUpdate() {
          // プログレスの値を更新する
          // Update the value of progress
          _distortion.setProgress(progress.value);
          current.el.style.opacity = progress.value;
        },
        onComplete() {
          world.hideMeshes(current.os);
          resolve();
        },
      });
    });
  }

  async function revealNext({ next }) {
    world.displayMeshes(next.os);

    // アニメーション用progress
    // Progress for animation
    const progress = { value: 0 };
    return new Promise((resolve) => {
      gsap.to(progress, {
        value: 1,
        duration: 0.5,
        delay: 0.3,
        ease: "power3.out",
        onUpdate() {
          // プログレスの値を更新する
          // Update the value of progress
          _distortion.setProgress(progress.value);
          next.el.style.opacity = progress.value;
        },
        onComplete() {
          // コンポーザからパスを削除
          // Remove the pass from the composer
          _distortion.removePass(false);
          resolve();
        },
      });
    });
  }

  // 各ページのenter, leaveを実行します
  // Execute enter and leave for each page
  function enterLeave(state) {
    const { current, next } = state;
    current.page.leave?.(state);
    next.page.enter?.(state);
  }
}

export { registTransitionDistortion };
