import { Texture } from "three";
export type Url = string;
export type DataKey = string;
/**
 * ローディングの進捗を受け取るコールバック関数です。ローディング時に画像などの読み込み状況に応じてアニメーションを付けたい場合に使用することができます。
 *
 * ```ts
 * import { loader } from "../negl";
 * // or
 * const { start } = setup();
 * const negl = await start((progress, total) => {
 *   console.log(`${progress}枚目の読み込みが完了。読み込み対象は[${total}]枚。`);
 * });
 * // > 1枚目の読み込みが完了。読み込み対象は3枚。
 * // > 2枚目の読み込みが完了。読み込み対象は3枚。
 * // > 3枚目の読み込みが完了。読み込み対象は3枚。
 * ```
 *
 * @param progress 読み込み完了したアセット（画像、動画）枚数
 * @param total 読み込み対象のアセット（画像、動画）枚数
 */
export type ProgressAction = (progress: number, total: number) => void;
/**
 * テクスチャマップ
 */
export type Texes = Map<string, Texture | undefined>;
/**
 * アセット（画像、動画）の読み込み管理を行います。
 *
 * {@link loader} 変数として negl からインポートして使用することができます。
 *
 * **取得方法**
 * ```ts
 * import { loader } from "../negl";
 * // or
 * window.negl.loader
 * ```
 *
 * @since 1.0.0
 */
export declare class Loader {
    #private;
    /**
     * 全てのアセットの読み込みが完了している場合 `true`, それ以外 `false`
     * @defaultValue `false`
     */
    isLoaded: boolean;
    /**
     * URLとそれに紐づくアセットから生成したテクスチャのマッピングを保持します。もし、URLでアセットが取得できなかった場合は`undefined`が格納されます。
     */
    textureCache: Map<string, Texture | undefined>;
    loadAllAssets(): Promise<void>;
    /**
     * HTMLに紐づくテクスチャを取得
     *
     * ```ts
     * const texes = await loader.getTexByElement(el);
     * ```
     * @param el HTML要素
     * @returns HTML要素に紐付く全てのテクスチャ
     */
    getTexByElement(el: HTMLElement): Promise<Texes>;
    /**
     * `url`に設定されたアセット（画像、動画）の読み込み、テクスチャを返却します。
     *
     * ```ts copy
     * const tex = await loader.loadTex('/img/sample.jpg');
     * ```
     * @param url
     * @returns
     */
    loadTex(url: Url): Promise<Texture | undefined>;
    /**
     * 読み込み進捗表示の関数追加メソッド
     * @internal
     */
    addProgressAction(action: ProgressAction): void;
}
/**
 * アセット（画像、動画）の読み込み管理を行う{@link Loader}オブジェクトです。
 *
 * 新しく画像や動画をテクスチャとして{@link Loader#loadTex 読み込む}ことができます。
 */
declare const loader: Loader;
export { loader };
//# sourceMappingURL=loader.d.ts.map