/**
 * three.js(WebGL)制御用ファイル
 */
import { Object3D } from 'three';
import { Mouse } from '../component';
import { Ob, World, ObOrSelector } from '../glsl';
import { GoToArgs } from '../component/transition';
export type MeshUIActionArgs = {
    status: 'enter' | 'leave' | 'click';
    mesh?: Object3D;
};
export type MeshUIAction = (args: MeshUIActionArgs) => void;
export type PageToArgs = Omit<GoToArgs, 'event' | 'triggerType'>;
export type PageTo = (args: PageToArgs) => void;
/**
 * メッシュに対するマウスのユーザーインターフェイス(UI)を制御する{@link MeshUI}クラスのオブジェクトです。
 *
 * マウスのクリックやホバーの状態を受け取りたいメッシュを監視対象として{@link MeshUI#watch 追加}、{@link MeshUI#unwatch 削除}したり、メッシュに対するクリックイベントの{@link MeshUI#stopPropagation 伝播の停止}や{@link MeshUI#preventDefault デフォルト制御の停止}などを行うことができます。
 *
 * [注意]
 * メッシュへのインタラクション制御は {@link hook} にコールバック関数を登録することで実装できます。その為、このオブジェクト自体を使用する機会は極めて稀です。
 *
 * **取得方法**
 * ```ts copy
 * import { meshUI } from "../negl";
 * // or
 * window.negl.meshUI
 * ```
 *
 * 監視対象に追加したメッシュにマウスが作用すると以下の処理が実行されます。
 * - 以下のフックの発火
 *   MOUSE_MESH_ENTER、MOUSE_MESH_LEAVE、MOUSE_MESH_CLICK
 *
 * **注意**
 * `data-interactive`を付与したものは自動的にUIの監視対象になります。詳細は{@link MeshUI}を確認してください。
 * ```html
 * <!-- 以下のHTMLに紐づくエフェクトは自動的に監視対象 -->
 * <div data-interactive data-webgl="some-effect"></div>
 * ```
 */
declare let meshUI: MeshUI;
/**
 * メッシュに対するマウスのインタラクションを制御します。
 * **{@link Hook} の裏で動いているクラスのため、基本的には直接使用しません。**
 *
 * このクラスは`negl`内部でインスタンス化され、{@link meshUI}オブジェクトとして提供されます。
 *
 * **取得方法**
 * ```ts
 * import { meshUI } from "../negl";
 * // or
 * window.negl.meshUI
 * ```
 */
declare class MeshUI {
    #private;
    static init(world: World, mouse: Mouse): Promise<MeshUI>;
    constructor(world: World, mouse: Mouse);
    watchMarks(os: Ob[]): void;
    /**
     * {@link Ob} オブジェクトに含まれる [Mesh](https://threejs.org/docs/#api/en/objects/Mesh) のUI監視対象に追加します。
     *
     * ```ts
     * // メッシュをマウスの監視対象に追加
     * meshUI.watch(o);
     * meshUI.watch('.css-selector');
     * ```
     *
     * **注意**
     * `data-interactive`を付与したものは自動的にUIの監視対象になります。
     * ```html
     * <!-- 以下のHTMLに紐づくエフェクトは自動的に監視対象 -->
     * <div data-interactive data-webgl="some-effect"></div>
     * ```
     *
     * @param obOrSelector {@link Ob} または セレクタ文字列
     * @returns
     */
    watch(obOrSelector: ObOrSelector): void;
    /**
     * マウスのUIに関する情報を初期化します。
     *
     * ```ts
     * // OBクラスに含まれるメッシュを監視対象から除外
     * meshUI.resetUIData(mesh);
     * ```
     */
    resetUIData(mesh: Object3D): void;
    /**
     * メッシュを監視対象から除外します。
     *
     * ```ts
     * // OBクラスに含まれるメッシュを監視対象から除外
     * meshUI.unwatch(o);
     * // or
     * meshUI.unwatch('.css-selector');
     * // or
     * meshUI.unwatch(domElement);
     * ```
     */
    unwatch(obOrSelector: ObOrSelector): void;
    /**
     * メッシュクリックイベント発火を一時的に停止します。
     *
     * 例えば、`button`要素とメッシュが重なって表示されている場合に`button`のクリックイベントのみを実行し、メッシュのクリックイベントを発火させたくない場合には次のように記述します。
     *
     * ```ts copy
     * buttonElement.addEventListener('click', () => {
     *  // ...
     *  // メッシュイベントの発火を停止
     *  meshUI.stopPropagation();
     * });
     * ```
     *
     * 以下のフックに登録された処理が停止対象となります。
     * - `hook.on(hook.MOUSE_MESH_CLICK, ...)`
     */
    stopPropagation(): void;
    /**
     * メッシュクリック時のデフォルト制御を一時的に停止します。
     * 例えば、`button`要素とメッシュが重なって表示されている場合に`button`のクリックイベントのみを実行し、メッシュのクリックイベントを発火させたくない場合には次のように記述します。
     *
     * ```ts copy
     * buttonElement.addEventListener('click', () => {
     *  // ...
     *  // メッシュのクリックイベントの発火を停止
     *  meshUI.stopPropagation();
     * });
     * ```
     *
     * 以下の処理が停止対象となります。
     * - HTML内の`data-click-1`によって登録されたアクション、および画面遷移
     * - リンククリックによる画面遷移
     */
    preventDefault(): void;
    /**
     * クリックのデフォルト制御
     * @internal
     * @param event
     * @returns
     */
    click(event: PointerEvent): false | undefined;
    /**
     * ホバー中のメッシュがあれば紐づく`Ob`オブジェクトを返却します。
     */
    getHoveringOb(): Ob | undefined;
    /**
     * ホバー中のメッシュがあれば対象のメッシュを返却します。
     * @returns [Obect3D](https://threejs.org/docs/#api/en/objects/Obect3D)
     */
    getHoveringMesh(): Object3D<import("three").Event> | undefined;
    /**
     * 監視対象の初期化を行います。
     */
    reset(): void;
}
export { meshUI, MeshUI };
//# sourceMappingURL=mesh-ui.d.ts.map