import { World } from "./glsl/world";
import { Viewport, Gui, INode, Config } from "./helper";
import { Transition, Scroller, Mouse, Loader, MeshUI, ProgressAction } from "./component";
export type NeglOptions = Partial<Config>;
/**
 * `negl` ライブラリのオブジェクトです。
 */
declare let negl: Negl;
/**
 * `negl` ライブラリの設定を行います。`negl` ライブラリを使用する際に一番最初に呼ぶ関数です。
 * @param options {@link config}
 * @returns `start` メソッドを含むオブジェクトを返却します。start メソッドを実行するとneglの初期化処理が実行されます。また、startの引数に渡したコールバック関数（[ProgressAction](/reference/modules/#progressaction)）では画像の読み込み進捗を受け取ることができます。
 */
declare function setup(options?: NeglOptions): {
    start: typeof start;
    hook: import("./helper/hook").Hook;
};
/**
 * `negl`ライブラリの初期化を行います。この関数が実行された後に`negl`の部品（world, scroller, mouseなど）が使用可能な状態になります。
 * @param progressAction {@link ProgressAction}
 */
export declare function start(progressAction: ProgressAction): Promise<Negl | undefined>;
/**
 * `negl`のコンポーネント（world、scrollerなど）へのアクセスを提供します。
 *
 * **取得方法**
 * ```ts
 * import { negl } from "./negl";
 * // or
 * window.negl
 * ```
 *
 * **前提条件**
 * `negl`へのアクセスは`setup`、`start`の実行後に行なってください。
 * ```ts
 * import { setup } from "./negl";
 * (async () => {
 *   const { start } = await setup();
 *   const negl = await start();
 *   // 以下、start実行後 `negl` 内のコンポーネントにアクセス可能
 *   // アクセス例）
 *   // 1. neglのプロパティから`mouse`コンポーネントにアクセス
 *   console.log(negl.mouse.getClipPos());
 * })();
 *
 * // 2. start() 実行後は静的インポートでも `negl` 内のコンポーネントにアクセス可能です。
 * // import { mouse } from "./negl";
 * ```
 */
declare class Negl {
    #private;
    $: {
        canvas: HTMLCanvasElement;
        pageContainer: HTMLElement;
        globalContainer: HTMLElement;
    };
    /**
     * {@link Hook}オブジェクトへのアクセスを提供します。
     */
    hook: import("./helper/hook").Hook;
    /**
     * {@link Config}オブジェクトへのアクセスを提供します。
     */
    config: Config;
    /**
     * {@link utils}オブジェクトへのアクセスを提供します。
     */
    utils: Record<string, any>;
    /**
     * {@link INode}オブジェクトへのアクセスを提供します。
     */
    INode: typeof INode;
    /**
     * 自オブジェクト（Neglオブジェクト）を返します。以下のような書き方が可能です。
     *
     * ```js
     * const { negl } = negl;
     * ```
     */
    negl: Negl;
    /**
     * {@link World}オブジェクトへのアクセスを提供します。 {@link start}関数の実行後にアクセス可能です。
     */
    world?: World;
    /**
     * {@link Viewport}オブジェクトへのアクセスを提供します。 {@link start}関数の実行後にアクセス可能です。
     */
    viewport?: Viewport;
    /**
     * {@link Scroller}オブジェクトへのアクセスを提供します。 {@link start}関数の実行後にアクセス可能です。
     */
    scroller?: Scroller;
    /**
     * {@link Mouse}オブジェクトへのアクセスを提供します。 {@link start}関数の実行後にアクセス可能です。
     */
    mouse?: Mouse;
    /**
     * {@link Transition}オブジェクトへのアクセスを提供します。 {@link start}関数の実行後にアクセス可能です。
     */
    transition?: Transition;
    /**
     * {@link Loader}オブジェクトへのアクセスを提供します。 {@link start}関数の実行後にアクセス可能です。
     */
    loader?: Loader;
    /**
     * {@link Gui}オブジェクトへのアクセスを提供します。 {@link start}関数の実行後にアクセス可能です。
     */
    gui?: Gui;
    /**
     * {@link MeshUI}オブジェクトへのアクセスを提供します。 {@link start}関数の実行後にアクセス可能です。
     */
    meshUI?: MeshUI;
    /**
     * 初期化処理
     * @internal
     * @returns
     */
    static init(options: NeglOptions): Negl;
    protected constructor(options?: NeglOptions);
    /**
     * 初期化処理
     * @internal
     * @returns
     */
    init(): Promise<this | undefined>;
    /**
     * WebGLのレンダリングを開始します。three.js のシーンのレンダリングを開始したいタイミングで実行してください。
     */
    renderStart(): void;
}
export { negl, Negl, setup };
//# sourceMappingURL=negl.d.ts.map