// スタイルの読み込み
// Loading styles
import "../styles/style.scss";

// ライブラリ
// Libraries
import lilGUI from "lil-gui";

// neglの読み込み
// Loading negl
import { setup, utils } from "negl";

/**
 * NOTE 以下使用は任意。なくても negl 自体は問題なく稼働。
 * NOTE The following usage is optional. negl itself will work fine without them.
 */
// GPUパフォーマンスによる制御のための準備関数
// Preparation functions for control based on GPU performance
import {
  definePerformanceMode,
  isLowPerformanceMode,
} from "#/parts/helper/detectGPU";
import { performanceMonitor } from "#/parts/helper/performance-monitor";
import { reloadOnSafari } from "#/parts/helper/safariBlurReload";

// ローディングアニメーション制御
// Loading animation control
import {
  letsBegin,
  progressCounter,
  addLoadingAnimation,
} from "#/parts/loader-travel";

// マウスカーソル制御
// Mouse cursor control
import initMousePic from "#/parts/mouse/mouse-pic";
import { initMouseMeshPointer } from "#/parts/mouse/mouse-mesh-pointer";

// ページトランジション用制御
// Control for page transitions
import { registTransitionBase } from "#/parts/transition/transition-base";
import { registTransitionStream } from "#/parts/transition/transition-stream";
import { registTransitionSlotPos } from "#/parts/transition/transition-slot-pos";

// デバッグモードを安全に制御する関数
// Function to safely control debug mode
const enableDebugMode = (debug) => {
  // viteが開発か本番かによって値を設定してくれる。
  // vite sets the value depending on whether it's development or production.
  return debug && import.meta.env.DEV;
};

// ページの読み込み
// Loading pages
const pageResolver = (type) => import(`./page/${type}.js`);

// OBの読み込み
// Loading OB
const obResolver = async (type) => {
  const module = await import(`./parts/glsl/${type}/index.js`);
  return module.default;
};

(async () => {
  // パフォーマンスモードの初期化
  // Initializing performance mode
  await definePerformanceMode(2, 35);

  // デバイスによってテクスチャを変更
  // パフォーマンスが低いデバイスの際は小さな画像を設定
  // Change textures depending on the device
  // Set smaller images for devices with low performance
  updateTexImageSize();

  // デバッグモード：1, 非デバッグモード：0
  // Debug mode: 1, Non-debug mode: 0
  const debug = enableDebugMode(1);
  // neglの設定 *必ず一番最初に呼び出す。
  // 参考：[neglドキュメント] -　https://docs.not-equal.dev/tutorial/installation/#negl-にオプションを渡してみよう
  // negl configuration *Always call this first.
  // Reference: [negl Documentation] - https://docs.not-equal.dev/tutorial/installation/#negl-にオプションを渡してみよう
  const { start } = setup({
    resolver: {
      page: pageResolver,
      ob: obResolver,
    },
    viewport: {
      resizeDuration: 0.5,
      near: 500,
      devicePixelRatio: utils.isTouchDevices() ? 2 : 1,
    },
    scroller: {
      damp: 3,
    },
    transition: { defaultType: "stream", spaMode: true },
    addon: {
      lilGUI,
    },
    world: {
      clearColor: "#ffffff",
      // clearAlpha: 0,
    },
    debug,
  });

  /**
   * neglの初期化
   * 以降、negl内の全てのオブジェクトにアクセス可能。
   *
   * Initialization of negl
   * From this point on, all objects within negl are accessible.
   */
  const negl = await start(progressCounter);
  const { hook } = negl;

  // デバイスに合わせて画像サイズを設定
  // Set image sizes according to the device
  hook.on(hook.T_NEXT_DOM_BEFORE, updateTexImageSize);

  // ページトランジションの追加
  // Adding page transitions
  defineTransition(negl);
  registTransitionBase(negl);
  registTransitionSlotPos(negl);

  // 画面遷移のときのシェーダーパスの設定を登録
  // Register shader path settings for screen transitions
  const streamPass = await registTransitionStream(negl);
  streamPass.addPass(); // シェーダーパスを追加. Add a shader path.
  // シェーダーパスのuniformsの値を設定
  // Set the values of the shader pass uniforms
  streamPass.setProgress(0.3);
  streamPass.setScale(-7);
  streamPass.setAlpha(0);

  /**
   * NOTE requestAnimationFrameによるシーンのレンダリング開始！
   * シーンを表示する直前に描写を開始
   * NOTE Start rendering the scene with requestAnimationFrame!
   * Begin drawing just before displaying the scene.
   */
  negl.renderStart();

  // Webページのパフォーマンス(FPS)の測定の設定
  // Setting up performance (FPS) measurement for the web page
  performanceMonitor(negl);

  // マウス制御
  // Mouse control
  const img = "/img/gallery/hokkaido-700.jpg";
  initMousePic(negl, img);
  initMouseMeshPointer(negl);

  /**
   * ローディング完了時のアニメーション
   * Animation upon loading completion
   */
  function loadAnimation(tl) {
    const progress = { value: 0 };
    tl.to(progress, {
      value: 0.3,
      // 表示タイミングを少しランダムにする
      // Slightly randomize the timing of the display
      duration: 1 + Math.random(),
      ease: "power0",
      onUpdate() {
        // パスの透明度を少しずつ変更。これが 1 になるとメッシュが表示される
        // Gradually change the opacity of the path. When this becomes 1, the mesh is displayed
        streamPass.setAlpha(progress.value);
      },
    }).to(progress, {
      value: 1,
      duration: 1,
      ease: "expo.inout",
      onUpdate() {
        // パスの透明度を少しずつ変更
        // Gradually change the opacity of the path
        streamPass.setAlpha(progress.value);
        // パスの形状を少しずつ変更
        // Gradually change the shape of the path
        streamPass.setProgress(progress.value);
        streamPass.setScale((progress.value - 1) * 4);
      },
      onComplete() {
        // アニメーションが完了した時点でパスをシーンから除去
        // Remove the path from the scene once the animation is complete
        streamPass.removePass(false);
      },
    });
  }

  // safariで残像が見える現象への対策
  // Countermeasure for afterimages seen in Safari
  reloadOnSafari(negl);

  // ローディングアニメーションの登録
  // Registering the loading animation
  addLoadingAnimation(loadAnimation);

  /**
   * ローディング完了時のアニメーションの開始
   * あくまでユーザーの独自関数のため、ライブラリの機能ではありません。
   *
   * Start of the animation upon loading completion
   * This is purely a user's custom function and not a feature of the library.
   */
  await letsBegin();
})();

// 遷移先がmenuページかどうかの判定
// Determine if the transition destination is a menu page
function isNextMenuPage(next) {
  return next.url.indexOf("menu") !== -1;
}

/**
 * 画面遷移の定義
 * Definition of screen transitions
 */
function defineTransition({ transition }) {
  // 参考：[neglドキュメント] - https://docs.not-equal.dev/tutorial/transition/
  // Reference: [negl Documentation] - https://docs.not-equal.dev/tutorial/transition/
  transition.changeType((transitionType, { current, next, triggerType }) => {
    // トップページからの遷移の場合は transitionType をそのまま返す
    // Return the transitionType as is for transitions from the top page
    const title = document.querySelector(".fv__title");
    if (title?.textContent === "Beautiful Japan") {
      return transitionType;
    }

    // 参考：[neglドキュメント] - https://docs.not-equal.dev/reference/interfaces/TransitionState/#triggertype
    // Reference: [negl Documentation] - https://docs.not-equal.dev/reference/interfaces/TransitionState/#triggertype
    switch (triggerType) {
      case "mesh":
      case "link":
        if (current.pageType === "menu" && !isNextMenuPage(next)) {
          return "slot-pos";
        }
        if (current.pageType === "detail" && isNextMenuPage(next)) {
          return "slot-pos";
        }
      case "popstate":
        if (current.pageType === "detail" && isNextMenuPage(next)) {
          return "slot-pos";
        }
    }
    return transitionType;
  });
}

/**
 * デバイスに合せて表示する画像のサイズを変更（パフォーマンス対応）
 * Change the size of the images displayed according to the device (performance adaptation)
 */
function updateTexImageSize({ next } = {}) {
  const scope = next?.el ?? document;
  const regex = /^\/img\/[^-]+\.jpg$/;

  const els = scope.querySelectorAll('[data-tex-1$=".jpg"]');
  // タッチデバイス または ローパフォーマンスモードの時
  // When it's a touch device or in low performance mode
  if (utils.isTouchDevices() || isLowPerformanceMode()) {
    // 700の小さい画像を使用
    // Use the smaller 700px images
    els.forEach((el) => {
      for (let key in el.dataset) {
        const value = el.dataset[key];
        if (regex.test(value)) {
          // imgのパスの末尾に-700を付与
          // Append -700 to the end of the img path
          const smallImg = value.replace(/\.jpg$/, "-700.jpg");
          el.dataset[key] = smallImg;
        }
      }
    });
  }
}
