/**
 * メニュー画面（menu.html）用制御ファイル
 * Control file for the menu screen (menu.html)
 *
 * data-page="menu" ページ独自の処理を行います。
 * Implements page-specific processing for data-page="menu".
 */
import { negl, world, viewport } from "negl";
import { mouseCamera } from "#/parts/helper/mouseCamera";
import { Vector3 } from "three";
import { initMenu } from "#/parts/initMenu";

export default async function () {
  // シリンダー（slider-slot）メッシュの初期化
  // Initialize the cylinder (slider-slot) mesh
  initMenu(negl);

  const { camera } = world;

  // カメラの向きを原点に設定
  // Set the camera's direction to the origin
  const lookAt = new Vector3(0, 0, 0);

  // マウスカーソルの位置によってカメラの位置を動かします。詳細は mouseCamera.ts を確認してください。
  // Move the camera position based on the mouse cursor location. For details, please check mouseCamera.ts.
  mouseCamera(
    // Negl オブジェクトを渡します。
    // Pass the Negl object.
    negl,
    // マウスが動いた際のカメラの位置を { x, y, z } のオブジェクト形式で返却する関数を渡します。
    // Pass a function that returns the camera position as an { x, y, z } object when the mouse moves.
    ({ x, y }) => ({
      x: x * 400,
      y: -y * 500,
      z:
        2000 -
        700 * Math.sin((x * Math.PI) / 2) ** 2 -
        (300 * Math.cos(y * Math.PI)) / 2,
    }),
    // カメラの向きを定義した Vector3 のオブジェクトを渡します。
    // Pass the Vector3 object that defines the camera's direction.
    lookAt,
    // ページ遷移の際にカメラのマウスへの追従をストップするための関数を渡します。この関数で true が返ったとき、マウスへの追従は停止します。また、 targetPosition の値を変更するとその位置に向かってカメラが移動します。
    // Pass a function to stop the camera from following the mouse during page transitions. When this function returns true, the following of the mouse will stop. Also, changing the value of targetPosition will move the camera towards that position.
    function destroy(targetPosition) {
      // マウスを停止させたい場所を設定します。
      // Set the location where you want to stop the mouse.
      targetPosition.x = 0;
      targetPosition.y = 0;
      targetPosition.z = viewport.cameraZ;

      // 現在のカメラの位置が目的地（targetPosition）に十分近づいた時、true を返します。（マウスへの追従を停止します。）
      // Returns true when the current camera position is sufficiently close to the destination (targetPosition). (This stops the following of the mouse.)
      return (
        Math.abs(camera.position.x) < 1e-5 &&
        Math.abs(camera.position.y) < 1e-5 &&
        Math.abs(camera.position.z - viewport.cameraZ) < 1e-5
      );
    }
  );
}
