import {
  CylinderGeometry,
  MeshBasicMaterial,
  Mesh,
  PlaneGeometry,
  Quaternion,
  ShaderMaterial,
  Vector3,
  DoubleSide,
  Spherical,
  UniformsUtils,
  MathUtils,
} from "three";
import gsap from "gsap";
import vertexShader from "./vertex.glsl";
import fragmentShader from "./fragment.glsl";
import textVertexShader from "./textVertex.glsl";
import textFragmentShader from "./textFragment.glsl";
import { isMobile } from "#/parts/helper/isMobile";
import { utils, INode, viewport } from "negl";
import { MultiMeshSlider } from "#/parts/helper/slider/MultiMeshSlider";
export default class extends MultiMeshSlider {
  beforeCreateMesh() {
    super.beforeCreateMesh();

    this.radius = viewport.height * this.dataAttrs.radius; // 画面の高さに応じてシリンダーの半径を決める
    this.rotateAxis = new Vector3(1, 0, 0); // シリンダーの回転軸
    this.scaleOffset = { ...this.scale };
    this.visibleThreshold = isMobile() ? 0.95 : 0.8; // 各スライドを透明にする閾値
    this.offsetRad = MathUtils.degToRad(this.dataAttrs.offsetDeg) ?? 0; // シリンダーの初期回転角度
  }

  setupGeometry() {
    const wSeg = 10,
      hSeg = 10;
    // シリンダに貼り付けるスライドのジオメトリ
    const geometry = new PlaneGeometry(
      this.rect.width,
      this.rect.height,
      wSeg,
      hSeg
    );
    return geometry;
  }

  setupMaterial() {
    return new ShaderMaterial({
      defines: this.defines,
      vertexShader: this.vertexShader,
      fragmentShader: this.fragmentShader,
      uniforms: this.uniforms,
      transparent: true,
      onBeforeCompile: this.onBeforeCompile, // 2023/5/5 WebGL1.0対応
    });
  }

  setupVertex() {
    return vertexShader;
  }

  setupFragment() {
    return fragmentShader;
  }

  setupUniforms() {
    const uniforms = super.setupUniforms();
    uniforms.uDist = { value: 0.8 };
    uniforms.uRadius = { value: this.radius };

    // テクスチャは観光地画像と文字の画像が存在し、それらを１セットとしてカウントするため２で割る
    uniforms.uSlideTotal = { value: this.texes.size / 2 };
    uniforms.uCurve = { value: this.dataAttrs.curve ?? 1 };

    uniforms.uAlpha.value = 1;
    uniforms.uProgress.value = 0;

    return uniforms;
  }

  setupTexes(uniforms) {
    return uniforms;
  }

  setupMesh() {
    const cylinderGeo = new CylinderGeometry(
      this.radius,
      this.radius,
      this.rect.width, // 円柱の高さをHTML要素の横幅に設定。
      18,
      1,
      true
    );

    const cylinderMate = new MeshBasicMaterial({
      transparent: true,
      opacity: 0,
      alphaTest: 0.5,
      wireframe: true,
      color: 0xff0000,
    });

    const cylinder = new Mesh(cylinderGeo, cylinderMate);

    let idxP = 0;
    let idxT = 0;
    this.texts = []; // 文字の画像のメッシュ用
    this.slides = []; // 画像用

    this.texes.forEach((tex, key) => {
      // タイトル用のテクスチャの場合
      if (key.startsWith("texT")) {
        // 文字の画像メッシュの場合の処理
        const textGeo = new PlaneGeometry(this.rect.width, 150, 1, 10);

        const textMate = new ShaderMaterial({
          defines: this.defines,
          vertexShader: textVertexShader,
          fragmentShader: textFragmentShader,
          uniforms: UniformsUtils.clone(this.uniforms),
          transparent: true,
          onBeforeCompile: this.onBeforeCompile, // 2023/5/5 WebGL1.0対応
        });

        textMate.side = DoubleSide;
        textMate.uniforms.tex1 = { value: tex };
        textMate.uniforms.uSlideIdx.value = idxT;
        textMate.uniforms.uActiveSlideIdx = this.uniforms.uActiveSlideIdx;
        textMate.uniforms.uTick = this.uniforms.uTick;
        textMate.uniforms.uDist = this.uniforms.uDist;
        textMate.uniforms.uAlpha = this.uniforms.uAlpha;
        textMate.uniforms.uCurve = this.uniforms.uCurve;

        const text = new Mesh(textGeo, textMate);

        this.texts.push(text);
        cylinder.add(text);

        idxT++;
      } else {
        // 画像の場合の処理
        const planeMate = this.material.clone();
        planeMate.side = DoubleSide;
        planeMate.uniforms.tex1 = { value: tex };
        planeMate.uniforms.uSlideIdx.value = idxP;
        planeMate.uniforms.uActiveSlideIdx = this.uniforms.uActiveSlideIdx;
        planeMate.uniforms.uTick = this.uniforms.uTick;
        planeMate.uniforms.uDist = this.uniforms.uDist;
        planeMate.uniforms.uAlpha = this.uniforms.uAlpha;
        planeMate.uniforms.uCurve = this.uniforms.uCurve;

        const planeGeo = this.geometry;

        const plane = new Mesh(planeGeo, planeMate);

        this.slides.push(plane);
        cylinder.add(plane);

        idxP++;
      }
    });

    // シリンダー上面の法線を回転軸(x軸方向)に向ける
    // 参考：[レクチャー] - https://not-equal.teachable.com/courses/webgl-practical/lectures/46832527
    utils.pointTo(cylinder, cylinder.up, this.rotateAxis);

    return cylinder;
  }

  async resize(duration = 1) {
    this.resizing = true;

    const {
      $: { el },
      mesh: cylinder,
      originalRect,
    } = this;

    const nextRect = INode.getRect(el);
    const { x, y } = this.getWorldPosition(nextRect, viewport);

    const p1 = new Promise((onComplete) => {
      gsap.to(cylinder.position, {
        x,
        y,
        overwrite: true,
        duration,
        onComplete,
      });
    });

    // 大きさの変更
    // 参考：[レクチャー] - https://not-equal.teachable.com/courses/webgl-practical/lectures/46832547
    const { position, normal } = cylinder.geometry.attributes;
    const ONE_LOOP = cylinder.geometry.attributes.position.count / 2;
    const step = Math.floor(ONE_LOOP / this.slides.length);
    const p2 = new Promise((onComplete) => {
      gsap.to(this.scale, {
        width: nextRect.width / originalRect.width,
        height: nextRect.height / originalRect.height,
        depth: 1,
        overwrite: true,
        duration,
        onUpdate: () => {
          cylinder.scale.set(
            this.scale.width,
            this.scale.height,
            this.scale.width
          );

          for (let i = 0; i < this.slides.length; i++) {
            const pickIdx = i * step;

            this.slides[i].position.x = position.getX(pickIdx);
            this.slides[i].position.z = position.getZ(pickIdx);

            const originalDir = { x: 0, y: 0, z: 1 };
            const targetDir = {
              x: normal.getX(pickIdx),
              y: 0,
              z: normal.getZ(pickIdx),
            };

            // メッシュの向きを変える
            utils.pointTo(this.slides[i], originalDir, targetDir);

            // クォータニオンの作成
            const q = new Quaternion();
            q.setFromAxisAngle(new Vector3(0, 0, 1), Math.PI / 2);
            // メッシュを回転
            this.slides[i].quaternion.multiply(q);

            // 回転角(写真とテキストの間の角度)を計算
            const textRotAngle = Math.atan(
              (this.rect.height / this.radius) * 0.72
            );

            // 極座標生成（テキスト画像をシリンダの面に沿って移動させるため）
            const spCoord = new Spherical();
            spCoord.setFromCartesianCoords(
              this.slides[i].position.x,
              this.slides[i].position.y,
              this.slides[i].position.z
            );
            spCoord.theta += textRotAngle;
            const vector = new Vector3();

            // 極座標をデカルト座標に戻す
            vector.setFromSpherical(spCoord);

            // テキスト画像を移動
            this.texts[i].position.x = vector.x;
            this.texts[i].position.y = vector.y;
            this.texts[i].position.z = vector.z;

            // テキスト画像の法線の向きを変える
            utils.pointTo(this.texts[i], originalDir, vector);
            this.texts[i].quaternion.multiply(q); // テキスト画像を回転
          }

          this.radius = viewport.height * this.dataAttrs.radius;
          cylinder.position.z = -this.radius; // シリンダの側面を原点に移動
        },
        onComplete,
      });
    });

    // 画像の解像度の調整
    // テクスチャのアスペクト比を滑らかに変化させる
    const p3 = new Promise((onComplete) => {
      const resolution = this.getResolution(nextRect);
      gsap.to(this.uniforms.uResolution.value, {
        x: resolution.x,
        y: resolution.y,
        z: resolution.z,
        w: resolution.w,
        overwrite: true,
        duration,
        onComplete,
      });
    });

    await Promise.all([p1, p2, p3]);

    this.rect = nextRect;

    this.resizing = false;
  }

  render(tick) {
    super.render(tick);

    const idx = utils.lerp(
      this.easeSlideIdx, // 開始点
      this.activeSlideIdx, // 終点
      this.slideSpeed // 係数
    );

    if (this.activeSlideIdx === idx) return;

    this.rotate(idx - this.easeSlideIdx);

    this.setUActiveSlideIdx(idx);

    this.setVisible(idx);

    this.easeSlideIdx = idx;
  }

  rotate(idx) {
    const diffRad = -(idx / this.slides.length) * 2 * Math.PI;
    this.mesh.rotateOnWorldAxis(this.rotateAxis, diffRad);
  }

  setUActiveSlideIdx(idx) {
    this.uniforms.uActiveSlideIdx.value = idx;
  }

  setVisible(idx) {
    // 前面に出ているスライド以外を消す処理
    this.slides.forEach((slide, i) => {
      // メッシュの位置を表す関数を、cosの絶対値を取って正の値をとる周期関数にする。
      let meshPos = Math.abs(
        Math.cos(
          ((idx - i) / slide.material.uniforms.uSlideTotal.value) * Math.PI
        )
      );
      // メッシュの位置が閾値を下回ったら消す。
      if (meshPos < this.visibleThreshold) {
        slide.visible = false;
        this.texts[i].visible = false;
      } else {
        slide.visible = true;
        this.texts[i].visible = true;
      }
    });
  }

  setTo(idx, offsetRad = 0) {
    this.rotate(idx - offsetRad - this.activeSlideIdx);
    super.setTo(idx);
    this.setUActiveSlideIdx(idx);
    this.setVisible(idx);
  }

  afterInit() {
    this.texes.forEach((tex) => {
      tex.source.data.pause?.();
    });

    this.setTo(this.activeSlideIdx, this.offsetRad);
  }

  debug(folder) {
    super.debug(folder);

    folder.add(this.uniforms.uAlpha, "value", 0, 1, 0.1).name("alpha").listen();

    const changeRotateAxis = () => {
      utils.pointTo(this.mesh, this.mesh.up, this.rotateAxis.normalize());
    };

    folder.add(this.uniforms.uDist, "value", 0, 1, 0.01).name("uDist").listen();
    folder
      .add(this.rotateAxis, "x", -1, 1, 0.01)
      .name("rotation.x")
      .listen()
      .onChange(changeRotateAxis);
    folder
      .add(this.rotateAxis, "y", -1, 1, 0.01)
      .name("rotation.y")
      .listen()
      .onChange(changeRotateAxis);

    folder
      .add(this.rotateAxis, "z", -1, 1, 0.01)
      .name("rotation.z")
      .listen()
      .onChange(changeRotateAxis);

    folder
      .add(
        this.mesh.children[0].material.uniforms.uProgress,
        "value",
        0,
        1,
        0.1
      )
      .name("progess")
      .listen();

    const datData = { next: !!this.uniforms.uProgress.value };
    folder.add(datData, "next").onChange(() => {
      gsap.to(this.uniforms.uProgress, {
        value: +datData.next,
        duration: 2,
        ease: "power2.inOut",
      });
    });
  }
}
