import { scrollSnap } from "#/parts/helper/scrollSnap";
import { isMobile } from "#/parts/helper/isMobile";
import gsap from "gsap";
import { Group } from "three";

let _world,
  isCurrentMobile,
  mobileMenus, // モバイルメニューObオブジェクトの配列. Array of mobile menu Ob objects
  desktopMenus, // デスクトップメニューのObオブジェクトの配列. Array of desktop menu Ob objects
  mobileGroup, // 移動用：モバイルメニューメッシュグループ. For movement: Mobile menu mesh group
  desktopGroup; // 移動用：デスクトップメニューメッシュグループ. For movement: Desktop menu mesh group

export async function initMenu({ world, hook }) {
  initVariables(world, hook); // 上で定義している変数の初期化とsceneへの追加. Initialization of variables defined above and addition to scene
  fixMenu(); // シリンダーメッシュの位置を固定. Fix the position of the cylinder mesh
  enableMenuScroll(); // シリンダーメッシュのスナップする回転を可能にする処理. Process to enable snapping rotation of the cylinder mesh
  enableMenuResize(hook); // 画面幅をリサイズした時の処理を登録. Register the process when resizing the screen width
}

/**
 * 変数の初期化とシーンへの追加を行います。
 * Initializes variables and adds them to the scene.
 * @param {*} world
 * @param {*} hook
 */
function initVariables(world, hook) {
  _world = world;
  // モバイル用メニューとデスクトップ用メニュをそれぞれ取得します。
  // モバイル用メニューの取得（１つ）
  // Retrieve the mobile and desktop menus respectively.
  // Get the mobile menu (one)
  mobileMenus = world.getObByElAll(".menu-slider.mobile");
  // デスクトップ用メニュー（３つ）
  // Desktop menu (three)
  desktopMenus = world.getObByElAll(".menu-slider.desktop");

  // メッシュをグループ化
  // メッシュを動かしやすいようにグループ化します。
  // Group the meshes
  // Group the meshes to make them easier to move.
  mobileGroup = setupMeshGroup(mobileMenus);
  desktopGroup = setupMeshGroup(desktopMenus);

  // グループ化したものはシーンに追加しないと画面に表示されないためシーンに追加します。
  // Add the grouped items to the scene, otherwise they won't be displayed on the screen
  world.scene.add(mobileGroup);
  world.scene.add(desktopGroup);

  // 画面遷移時にシーンからグループを削除します。
  // Remove the group from the scene when transitioning to another screen.
  hook.on(hook.T_BOTH_OB_EXIST, removeGroup, { priority: 1500, once: true });
}

// 複数のシリンダーメッシュを固定するための関数
// Function to fix multiple cylinder meshes
function fixMenu() {
  mobileMenus.forEach((menu) => (menu.fixed = true));
  desktopMenus.forEach((menu) => (menu.fixed = true));
}

// sceneからグループ化したメッシュを削除。menu.htmlから遷移するときだけ実行
// Remove the grouped meshes from the scene. Execute only when transitioning from menu.html
function removeGroup({ next }) {
  if (next.pageType === "menu") return;
  _world.scene.remove(mobileGroup);
  _world.scene.remove(desktopGroup);
}

let resizeProgress = { value: 0 }; // メニュー切り替えの状態遷移（0:開始 1:終了）. State transition for menu switching (0: start, 1: end)
const distance = 1000; // モバイルメニューとデスクトップメニューを左右に配置する. Position the mobile and desktop menus side by side
function enableMenuResize(hook) {
  // 初期表示の際にどちらかを画面から非表示とする
  // 配置 mobile menu --> distance --> desktop menu

  // Hide one of them from the screen during the initial display
  // Position mobile menu --> distance --> desktop menu
  displayMenu();

  // デスクトップ、モバイルで表示を切り替える
  // Switch display between desktop and mobile
  isCurrentMobile = isMobile();
  // 画面幅をリサイズしたときのresizeProgress.valueの変更
  // Change resizeProgress.value when resizing the screen width
  const resize = () => {
    // 初期表示の時と異なるとき。
    // When different from the initial display.
    if (isCurrentMobile !== isMobile()) {
      resizeProgress.value = 0;

      gsap.to(resizeProgress, {
        value: 1,
        overwrite: true,
        onUpdate: isMobile() ? displayMobile : displayDesktop,
        onComplete() {},
      });

      isCurrentMobile = !isCurrentMobile;
    }
  };
  // 画面幅をリサイズしたときに発火させる関数を登録
  // Register the function to be fired when resizing the screen width
  hook.on(hook.RESIZE_DELAY, resize);
  // 画面遷移の時に発火する処理を登録
  // Register the process to be fired when transitioning to another screen
  hook.on(
    hook.T_BEGIN,
    () => {
      // 画面を離れる時トリガー削除（削除しないと、別ページでもメニューのresize処理が動きます。）
      // Remove the trigger when leaving the screen (if not removed, the menu resize process will work on other pages as well.)
      hook.off(hook.RESIZE_DELAY, resize);
    },
    { once: true }
  );
}

function setupMeshGroup(menuObs) {
  const group = new Group();
  menuObs.forEach((menu) => group.add(menu.mesh));
  return group;
}

function enableMenuScroll() {
  scrollSnap(
    (slideIdx) => desktopMenus[0].goTo(-slideIdx),
    {
      speed: 7.2,
      damping: 2.7,
    },
    -desktopMenus[0].activeSlideIdx
  );
  scrollSnap(
    (slideIdx) => desktopMenus[1]?.goTo(-slideIdx),
    {
      speed: 7.6,
      damping: 2.4,
    },
    -desktopMenus[1].activeSlideIdx
  );
  scrollSnap(
    (slideIdx) => desktopMenus[2]?.goTo(-slideIdx),
    {
      speed: 7,
      damping: 3,
    },
    -desktopMenus[2].activeSlideIdx
  );
  scrollSnap(
    (slideIdx) => mobileMenus[0]?.goTo(-slideIdx),
    {
      speed: 12,
      damping: 3,
    },
    -mobileMenus[0].activeSlideIdx
  );
}

// Mobile用にシリンダーメッシュの位置とスケールを調整
// Adjust the position and scale of the cylinder mesh for mobile
function displayMobile(progress = resizeProgress) {
  desktopGroup.position.x = distance * progress.value;
  desktopGroup.scale.setScalar(1 - progress.value);
  mobileGroup.position.x = -distance * (1 - progress.value);
  mobileGroup.scale.setScalar(progress.value);
}

// Desktop用にシリンダーメッシュの位置とスケールを調整
// Adjust the position and scale of the cylinder mesh for desktop
function displayDesktop(progress = resizeProgress) {
  desktopGroup.position.x = distance * (1 - progress.value);
  desktopGroup.scale.setScalar(progress.value);
  mobileGroup.position.x = -distance * progress.value;
  mobileGroup.scale.setScalar(1 - progress.value);
}

// シリンダーメッシュの出し分け処理
// Cylinder mesh display switching process
function displayMenu() {
  resizeProgress.value = 1;
  if (isMobile()) {
    // デスクトップメニューを右の画面外へ
    displayMobile(); // Move the desktop menu off the screen to the right
  } else {
    // モバイルメニューを左の画面外へ
    displayDesktop(); // Move the mobile menu off the screen to the left
  }
}
