﻿import gsap from "gsap";
import { Vector3 } from "three";
import { isMobile } from "#/parts/helper/isMobile";

import { getWorldCoordClonedMesh } from "#/parts/helper/getWorldCoordClonedMesh";

/**
 * シリンダーメッシュ（slider-slot）と下層ページのページトランジション処理をフックに登録します。
 * menu.html と kyoto.html などの下層ページ間のページトランジション処理を登録します。
 *
 * Registers the cylinder mesh (slider-slot) and the page transition process of the subpages to the hook.
 * Registers the page transition process between subpages such as menu.html and kyoto.html.
 */
export function registTransitionSlotPos({ hook, world }) {
  // トランジションタイプ "slot-pos"
  // Transition type "slot-pos"
  const type = "slot-pos";

  // トランジション開始
  // Start transition
  hook.on(hook.T_BEGIN, fadeOutCurrent, {
    type,
    priority: 700,
  });

  // トランジション 次のページのOBインスタンス挿入後
  // Transition after the OB instance of the next page is inserted
  hook.on(hook.T_BOTH_OB_EXIST, objTransition, { type, priority: 2000 });

  // 現在ページのアニメーション対象となるObオブジェクト
  // The Ob object that is the target of the animation for the current page
  let fromOb,
    // 次のページのアニメーション対象となるObオブジェクト
    // The Ob object that is the target of the animation for the next page
    toOb,
    // 下層ページの #main-img に紐づくメッシュ
    // The mesh associated with the #main-img of the subpage
    detailMainOb,
    // 現在ページのアニメーション対象となるメッシュ
    // The mesh that is the target of the animation for the current page
    fromMesh,
    // 次のページのアニメーション対象となるメッシュ
    // The mesh that is the target of the animation for the next page
    toMesh,
    isMenuToDetail,
    // アニメーションに使用するメッシュ
    // The mesh used for the animation
    transitionMesh,
    // アニメーションの目的地に存在するメッシュ
    // The mesh that exists at the destination of the animation
    destinationMesh,
    // 曲率
    // Curvature
    toCurve;

  // 遷移元のページの DOM をフェイドアウト
  // Fade out the DOM of the source page
  function fadeOutCurrent({ current }) {
    gsap.to(current.el, {
      opacity: 0,
    });
  }

  /**
   * menu.html（メニューページ）の slider-slot と 下層ページの #main-img 間でメッシュを移動します。
   *
   * 実装の流れとしては以下のようになっています。
   *
   * ## I. メニューページから下層ページへ遷移するとき
   *
   * 1. slider-slot 内のメッシュ（メッシュA）をクローン（アニメーションするメッシュに使用します。）
   * 2. メッシュA を #main-img までアニメーション
   * 3. メッシュA を破棄し、#main-img を表示
   *
   * ## II. 下層ページからメニューページへ遷移するとき
   *
   * 1. slider-slot 内のメッシュ（メッシュA）をクローン
   * 2. メッシュA の初期位置を #main-img に設定
   * 3. メッシュA を slider-slot の対象のメッシュ位置までアニメーション
   * 4. メッシュA を破棄し、slider-slot を表示
   *
   *
   * Moves the mesh between the slider-slot of menu.html (menu page) and the #main-img of the subpage.
   *
   * The implementation flow is as follows:
   *
   * ## I. When transitioning from the menu page to a subpage
   *
   * 1. Clone the mesh (Mesh A) inside the slider-slot (used for the animation mesh).
   * 2. Animate Mesh A to #main-img
   * 3. Dispose of Mesh A and display #main-img
   *
   * ## II. When transitioning from a subpage to the menu page
   *
   * 1. Clone the mesh (Mesh A) inside the slider-slot
   * 2. Set the initial position of Mesh A to #main-img
   * 3. Animate Mesh A to the target mesh position of the slider-slot
   * 4. Dispose of Mesh A and display the slider-slot
   *
   * @param [TransitionState](https://docs.not-equal.dev/reference/interfaces/TransitionState/)
   */
  async function objTransition(state) {
    // state - https://docs.not-equal.dev/reference/interfaces/TransitionState/
    const { next, current, triggerMesh } = state;

    // menu.html（メニューページ）から下層ページへの遷移の時 true。それ以外、false。
    // True when transitioning from menu.html (menu page) to a subpage, otherwise false.
    isMenuToDetail = current.pageType === "menu";

    // メニューページのスライダー（モバイル表示用、デスクトップ表示用）を取得します。
    // Retrieves the sliders for the menu page (for mobile and desktop display).
    const mobileMenu = world.getObByEl(".menu-slider.mobile");
    const desktopMenus = world.getObByElAll(".menu-slider.desktop");

    if (isMenuToDetail) {
      // メニューページから詳細ページへの遷移
      // Transition from the menu page to the detail page
      fromMesh = triggerMesh; // slider-slotが含むクリックされたメッシュ. The clicked mesh contained in the slider-slot
      fromOb = world.getObByMesh(fromMesh); // slider-slot

      /**
       * （ポイント）
       * メニューページで表示している slider-slot では各スライダー（plane）のメッシュを cylinder メッシュに追加した状態（slider-slot/index.js:L155）で保持しています。この状態では各 plane メッシュが保持する position は cylinder のメッシュのローカル座標となります。一方で、#main-img のメッシュ（normal-plane）はワールド座標に配置されています。
       *
       * その為、slider-slot の plane メッシュを #main-img の normal-plane の場所までアニメーションさせるためには plane メッシュをワールド座標系で動かす必要があります。
       *
       *
       * The key point:
       * In the menu page, the slider-slot holds each slider (plane) mesh added to the cylinder mesh (slider-slot/index.js:L155). In this state, the position held by each plane mesh becomes the local coordinates of the cylinder mesh. On the other hand, the mesh of #main-img (normal-plane) is placed in world coordinates.
       *
       * Therefore, to animate the plane mesh of the slider-slot to the location of the normal-plane of #main-img, it is necessary to move the plane mesh in the world coordinate system.
       */
      transitionMesh = getWorldCoordClonedMesh(fromMesh); // ワールド座標のクローンされたメッシュ. Cloned mesh in world coordinates

      // slider-slotのマテリアルをコピー
      // Copy the material of the slider-slot
      transitionMesh.material = transitionMesh.material.clone();

      // 下層ページの #main-img のエフェクト等を取得します。
      // Retrieves effects and such for the #main-img of the subpage.
      detailMainOb = toOb = world.getObByEl("#main-img");
      toMesh = toOb.mesh;

      // destinationMesh はアニメーションの目的地となるメッシュです。
      // destinationMesh is the mesh that is the destination of the animation.
      destinationMesh = getWorldCoordClonedMesh(toMesh); // ワールド座標のクローンされたメッシュ
      // The cloned mesh in world coordinates

      // 遷移先の曲率
      // The curvature of the destination
      toCurve = 0;
    } else {
      // 詳細ページからメニューページへの遷移
      // Transition from the detail page to the menu page
      detailMainOb = fromOb = world.getObByEl("#main-img");
      fromMesh = fromOb.mesh;
      transitionMesh = getWorldCoordClonedMesh(fromMesh); // ワールド座標のクローンされたメッシュ
      // The cloned mesh in world coordinates

      // 遷移先のメッシュ取得
      // Obtain the mesh of the destination
      const itemId = fromOb.dataAttrs["itemId"];
      if (isMobile()) {
        toOb = mobileMenu;
      } else {
        toOb = desktopMenus.find((o) => {
          // その他の場合はitemId内のitemIdが一致するslider-slotクラスを取得
          // In other cases, obtain the slider-slot class with a matching itemId within itemId
          return o.dataAttrs.itemId.split(",").includes(itemId);
        });
      }

      // 対象の画像に一致するスライド番号を取得
      // Obtain the slide number that matches the target image
      const sliderActiveIdx =
        toOb.dataAttrs.itemId.split(",").indexOf(itemId) % toOb.slides.length;
      toOb.setTo(sliderActiveIdx);
      toMesh = toOb.slides[sliderActiveIdx];
      destinationMesh = getWorldCoordClonedMesh(toMesh); // ワールド座標のクローンされたメッシュ. Cloned mesh in world coordinates

      /**
       * slider-slot のマテリアルを normal-plane のマテリアルとして設定
       *
       * （理由）
       * slider-slot のマテリアルを使わなければ、slider-slot と normal-plane（#main-imgのエフェクト）の両方の表示状態を再現できないためです。（normal-planeはメッシュを曲げて表示するなどの機能はありません。）
       *
       * Set the material of the slider-slot as the material for the normal-plane
       *
       * (Reason)
       * If the material of the slider-slot is not used, it is not possible to reproduce the display state of both the slider-slot and the normal-plane (the effect of #main-img). (The normal-plane does not have functions such as bending the mesh for display.)
       */
      transitionMesh.material = destinationMesh.material.clone();

      // 遷移先の曲率. The curvature of the destination
      toCurve = 1;
    }

    // アニメーションさせるメッシュをシーンに追加
    // Add the mesh to be animated to the scene
    world.scene.add(transitionMesh);

    /**
     * DOMから取得した DOMRect の比によって遷移元と遷移先のメッシュの大きさの比を求める。
     * Determine the ratio of the size of the mesh from the source to the destination by the ratio of the DOMRect obtained from the DOM.
     */
    const scale = new Vector3(
      toOb.rect.width / fromOb.rect.width,
      toOb.rect.height / fromOb.rect.height,
      1
    );

    // メニューページから下層ページへ遷移する際は下層ページメイン画像がリサイズされていないかを考慮する
    // Consider whether the main image of the subpage has been resized when transitioning from the menu page to the subpage
    if (!isMenuToDetail) {
      scale.x *= detailMainOb.scale.width;
      scale.y *= detailMainOb.scale.height;
    }

    const progress = { value: 0, toCurve: +!toCurve };

    // アニメーション開始位置や大きさ、角度等を変数に格納
    // Store the starting position, size, angle, etc., of the animation in variables
    const fromPosition = transitionMesh.position.clone(),
      fromScale = transitionMesh.scale.clone(),
      fromQuaternion = transitionMesh.quaternion.clone(),
      fromResolution = fromMesh.material.uniforms.uResolution.value.clone();

    /**
     * 後の gsap.to の onUpdate に設定します。
     * Set for the onUpdate of gsap.to later.
     */
    function onUpdate() {
      /**
       * メッシュの状態をA->Bへ遷移
       * Transition the state of the mesh from A to B
       */
      // メッシュのポジション
      // The position of the mesh
      // [lerpVectors](https://threejs.org/docs/#api/en/math/Vector3.lerpVectors)は 引数で渡された２つのベクトルの線形補間した値を設定します。
      // [lerpVectors](https://threejs.org/docs/#api/en/math/Vector3.lerpVectors) sets the linearly interpolated value of the two vectors passed as arguments.
      transitionMesh.position.lerpVectors(
        fromPosition,
        destinationMesh.position,
        progress.value
      );

      // メッシュのスケール
      // The scale of the mesh
      transitionMesh.scale.lerpVectors(fromScale, scale, progress.value);

      // メッシュの回転
      // The rotation of the mesh
      transitionMesh.quaternion.slerpQuaternions(
        fromQuaternion,
        destinationMesh.quaternion,
        progress.value
      );

      // テクスチャのアスペクト
      // The aspect of the texture
      transitionMesh.material.uniforms.uResolution.value.lerpVectors(
        fromResolution,
        destinationMesh.material.uniforms.uResolution.value,
        progress.value
      );

      /**
       * メッシュの曲率
       * scale.y を掛ける理由（geometry（元の平面の大きさ）が遷移元のメッシュと遷移先のメッシュでは異なるため、同じ uCurve を渡しても平面の曲がり方（曲率）が異なる。その為、scale.y で元のメッシュと遷移先のメッシュのY軸の大きさの比を掛けることで、差異を吸収します。）
       *
       * The curvature of the mesh
       * The reason for multiplying by scale.y (Because the geometry (the original size of the plane) is different between the mesh of the source and the destination, passing the same uCurve will result in different bending (curvature) of the plane. Therefore, by multiplying the ratio of the Y-axis size of the original mesh and the destination mesh by scale.y, the difference is absorbed.)
       */
      transitionMesh.material.uniforms.uCurve.value =
        progress.toCurve * scale.y;

      transitionMesh.material.uniforms.uProgress.value = progress.value;

      // 次のページの .page-container の透明度を 0 -> 1 に変化させます。
      // Changes the opacity of the .page-container of the next page from 0 to 1.
      next.el.style.opacity = progress.value;
    }

    await new Promise((resolve) => {
      gsap.to(progress, {
        value: 1,
        toCurve,
        duration: 1,
        ease: "power2.inout",
        onStart() {
          // アニメーション対象のクローンされたメッシュをシーンに追加（後に破棄）
          // Add the cloned mesh that is the subject of the animation to the scene (to be disposed of later)
          world.hideMeshes(current.os);
        },
        onUpdate,
        onComplete() {
          // シーンからアニメーション用のメッシュを削除します。
          // Remove the mesh for animation from the scene.
          world.scene.remove(transitionMesh);

          /**
           * 下層ページからメニューページに遷移する際は toCurve の情報をslider-slot の uniforms.uCurve の値として使用
           *
           * （理由）
           * 下層ページからメニューページに遷移する場合は geometry の情報は normal-plane（#main-imgのメッシュ）のものを使用しているため、そもそもの大きさ（mesh の scale を 1 とした際の大きさ）が遷移元と遷移先のメッシュで異なります。その為、同じ曲率（uCurve）を設定しても曲がる大きさが異なってくるため、アニメーション終了時点の progress.toCurve の値を slider-slot に設定し、平面の曲げ具合を保ちます。
           *
           *
           * When transitioning from a subpage to the menu page, use the information of toCurve as the value of uniforms.uCurve of the slider-slot
           *
           * The reason:
           * When transitioning from a subpage to the menu page, the geometry information used is that of the normal-plane (#main-img's mesh), so the original size (the size when the mesh's scale is set to 1) differs between the mesh of the source and the destination. Therefore, even if the same curvature (uCurve) is set, the degree of bending will differ, so the value of progress.toCurve at the end of the animation is set to the slider-slot to maintain the degree of bending of the plane.
           */

          if (!isMenuToDetail) {
            toMesh.material.uniforms.uCurve.value = progress.toCurve;
          }

          // アニメーション用メッシュに使用したマテリアルを破棄
          // Dispose of the material used for the animation mesh
          transitionMesh.material.dispose();

          // 次のページの Ob を全て表示
          // Display all Ob of the next page
          world.displayMeshes(next.os);

          // 次のフックの処理へ
          // Proceed to the next hook
          resolve();
        },
      });
    });
  }
}
