﻿import gsap from "gsap";
import initStreamPass from "#/parts/glsl/stream/pass";

/**
 * 画面を歪めアニメーションの画面遷移処理をフックに登録します。
 * Registers the screen distortion animation for screen transitions to the hook.
 */
async function registTransitionStream(negl) {
  // トランジションタイプ "stream"
  // Transition type "stream"
  const type = "stream";

  const { hook, world } = negl;

  // トランジション 初期時（1回のみ実行）
  // パスの初期化
  // Initialize the transition (executed only once)
  // Initialize the path
  let _stream = await initStreamPass(negl);
  _stream.setProgress(1);
  _stream.removePass(false);

  // トランジション 開始時
  // At the start of the transition
  hook.on(hook.T_BEGIN, fadeOutByStreamPass, { type, priority: 700 });

  // トランジション 次のページのOBインスタンス挿入後
  // After inserting the OB instance of the next page in the transition
  hook.on(hook.T_BOTH_OB_EXIST, revealNext, { type });

  // ディストーションのトランジション
  // Distortion transition
  const progress = { value: 1 };
  async function fadeOutByStreamPass({ current }) {
    _stream.addPass();

    progress.value = 1;
    return new Promise((resolve) => {
      gsap.to(progress, {
        value: 0,
        duration: 2,
        overwrite: true,
        ease: "power2.inout",
        onStart() {
          /**
           * パスによる歪み（模様）をランダムで設定します。
           * Sets a random distortion (pattern) to the path.
           */
          const rand = (Math.random() - 0.5) * 7;
          _stream.setScale(rand);
        },
        onUpdate() {
          // パスのアニメーションを更新します。
          // Update the path animation.
          _stream.setProgress(progress.value);

          // .page-container の透明度を変更します。
          // Change the opacity of .page-container.
          current.el.style.opacity = progress.value;

          // 現在ページの全てのエフェクトの透明度を 1 -> 0 に変更します。
          // Change the opacity of all effects on the current page from 1 to 0.
          current.os.forEach((o) => {
            o.uniforms.uAlpha.value = progress.value;
          });
        },
        onComplete() {
          // アニメーション完了後、現在ページのメッシュを全て非表示にします。
          // After the animation completes, hide all meshes of the current page.
          world.hideMeshes(current.os);

          // 元のページに戻ってきた時のために、uAlpha は 1 に戻しておきます。
          // Reset uAlpha to 1 for when returning to the original page.
          current.os.forEach((o) => {
            o.uniforms.uAlpha.value = 1;
          });
          resolve();
        },
      });
    });
  }

  /**
   * 遷移先の画面でのメッシュの表示とuniformsの変更
   * Displaying meshes and changing uniforms on the transition destination screen.
   * @param [TransitionState](https://docs.not-equal.dev/reference/interfaces/TransitionState/)
   * @returns
   */
  async function revealNext({ next }) {
    // uAlphaの初期値を0
    // Initialize uAlpha to 0
    next.os.forEach((o) => {
      o.uniforms.uAlpha.value = 0;
    });

    /**
     * next.os の メッシュの初期状態では mesh の visible は全て false のため、画面表示するためには visible = true に変更する。
     * Since the initial state of the meshes in next.os is all false for mesh visibility, change visible to true to display them on the screen.
     *
     * next.os - https://docs.not-equal.dev/tutorial/transition/#t_both_ob_exist次のページの-ob-オブジェクトがアクセス可能な状態
     * world.displayMeshes - https://docs.not-equal.dev/reference/classes/World/#displaymeshes
     */
    world.displayMeshes(next.os);

    // アニメーション用progress
    // Progress for animation
    return new Promise((resolve) => {
      gsap.to(progress, {
        value: 1,
        duration: 0.75,
        overwrite: true,
        ease: "expo.inout",
        onStart() {
          const rand = 3 + Math.random() * 3;
          _stream.setScale(rand);
        },
        onUpdate() {
          // パスのアニメーションを更新する。
          // Update the path animation.
          _stream.setProgress(progress.value);
          // 次のページの .page-container の透明度を 0 -> 1 に変更する。
          // Change the opacity of the .page-container of the next page from 0 to 1.
          next.el.style.opacity = progress.value;

          // 次のページのエフェクトの透明度を 0 -> 1 に変更する。
          // Change the opacity of the effects on the next page from 0 to 1.
          next.os.forEach((o) => {
            o.uniforms.uAlpha.value = progress.value;
          });
        },
        onComplete() {
          // パスを一時的に削除する。
          // Temporarily remove the path.
          _stream.removePass(false);
          resolve();
        },
      });
    });
  }

  return _stream;
}

export { registTransitionStream };
