/**
 * DOMアクセスのアダプタです。ライブラリ内で使用しているものになりますが、皆さんのプロジェクトではDOMに直接アクセス（document.querySelectorなど）していただいて大丈夫です！応用編で紹介しましたので、一応使えるようにライブラリでも提供しています。
 *
 * **アダプタとは？**
 * ブラウザのバージョンアップなどでDOMへのアクセス方法などが変わったときに備えて、ブラウザのネイティブの機能などを特定の関数を通して操作するデザインパターンです。
 */
declare class INode {
    /**
     * [querySelector](https://developer.mozilla.org/ja/docs/Web/API/Document/querySelector)のアダプタ
     * @param selector セレクタ文字列
     * @param scope querySelectorの検索範囲
     * @returns
     */
    static qs(selector: string, scope?: Document | Element): Element | null;
    /**
     * [querySelectorAll](https://developer.mozilla.org/ja/docs/Web/API/Document/querySelectorAll)のアダプタ
     * @param selector セレクタ文字列
     * @param { Document | Element } [scope=document] querySelectorAllの検索範囲
     * @returns
     */
    static qsAll(selector: string, scope?: Document | Element): Element[];
    /**
     * 引数で渡されたHTML文字列をDOMにして返却します。
     * @param selector HTML文字列
     * @returns HTML文字列から変換されたDOM
     */
    static htmlToEl(htmlStr: string): Element;
    /**
     * [Element](https://developer.mozilla.org/ja/docs/Web/API/Element)の場合 true が返ります。それ以外、false。
     */
    static isElement(target: unknown): target is Element;
    /**
     * セレクタ文字列、またはDOMを受け取り、DOMを返却します。
     * @param selector セレクタ文字列
     * @param scope querySelectorの検索範囲
     */
    static getElement(elementOrSelector: string | Element, scope?: Document | Element): Element | null;
    /**
     * セレクタ文字列、またはDOMに対応する[DOMRect](https://developer.mozilla.org/ja/docs/Web/API/DOMRect)を返却します。
     * @param elementOrSelector セレクタ文字列、またはDOM
     */
    static getRect(elementOrSelector: string | Element): DOMRect | null;
    /**
     * 対象のDOMのキーに対応するdatasetの値を取得します。
     * @param elementOrSelector セレクタ文字列、またはDOM
     * @param key datasetのキー
     */
    static getDS(elementOrSelector: string | Element, key: string): string | undefined;
    /**
     * 対象のDOMのdatasetにキーが存在する場合、trueを返します。それ以外の場合、false。
     * @param elementOrSelector セレクタ文字列、またはDOM
     * @param key datasetのキー
     */
    static hasDS(elementOrSelector: string | Element, key: string): boolean;
    /**
     * [addEventListener](https://developer.mozilla.org/ja/docs/Web/API/EventTarget/addEventListener)のアダプタ
     * @param elementOrSelector セレクタ文字列、またはDOM
     * @param type イベントタイプ
     * @param fn コールバック関数
     * @param options イベントリスナのオプション
     */
    static addListener(elementOrSelector: string | Element, type: string, fn: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;
    /**
     * [removeEventListener](https://developer.mozilla.org/ja/docs/Web/API/EventTarget/removeEventListener)のアダプタ
     * @param elementOrSelector セレクタ文字列、またはDOM
     * @param type イベントタイプ
     * @param fn イベント削除するコールバック関数
     * @param options イベントリスナのオプション
     */
    static removeListener(elementOrSelector: string | Element, type: string, fn: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;
}
export { INode };
//# sourceMappingURL=INode.d.ts.map