import { Object3D } from "three";
import { MeshUI, Mouse, Scroller, TransitionState } from "../component";
import { Ob, World } from "../glsl";
import { Viewport } from "./viewport";
import { Negl } from "..";
/**
 * フックオプション
 * {@link Hook.on Hook.on} の第三引数のオプションです。
 */
export type HookOptions = {
    /**
     * フックを実行する優先度
     * 同じフックタイプに複数のフックが登録されている場合は大きな数値のフックから実行されていきます。
     */
    priority?: number;
    /**
     * トランジションタイプ
     * トランジションタイプに一致するトランジションが実行されます。
     * 詳細は [トランジション](/tutorial/transition/) をご覧ください。
     */
    type?: string;
    /**
     * `true` の時はフックが一度実行されるとそのフックは解除されます。
     */
    once?: boolean;
};
/**
 * フック用コールバック
 * @internal
 */
export type HookCallback<T = any> = (args: T) => Promise<void> | void;
declare class HookType {
    /**
     * ページ毎のデフォルト関数実行前
     * @internal
     */
    readonly BEFORE_PAGE_INIT = 90;
    /**
     * マウスが動いた時
     * @internal
     */
    readonly MOUSEMOVE = 100;
    /**
     * マウスがメッシュに入った時
     * @internal
     */
    readonly MOUSE_MESH_ENTER = 200;
    /**
     * マウスがホバー中
     * @internal
     */
    readonly MOUSE_MESH_HOVER = 210;
    /**
     * マウスがメッシュから出た時
     * @internal
     */
    readonly MOUSE_MESH_LEAVE = 220;
    /**
     * マウスがメッシュをクリックした時
     * @internal
     */
    readonly MOUSE_MESH_CLICK = 230;
    /**
     * スクロールしている時
     * @internal
     */
    readonly SCROLL = 300;
    /**
     * フレーム毎の実行時
     * @internal
     */
    readonly RENDER = 400;
    /**
     * フレーム毎の実行時（必ず実行したいもの）
     * @internal
     */
    /**
     * 画面サイズが変わった時
     * @internal
     */
    readonly RESIZE = 500;
    /**
     * 画面サイズの更新が終了した時
     * @internal
     */
    readonly RESIZE_DELAY = 510;
    /**
     * ページトランジションの初期化時
     * @internal
     */
    readonly T_INIT = 610;
    /**
     * ページトランジションの開始時
     * @internal
     */
    readonly T_BEGIN = 620;
    /**
     * 次のページのDOMが挿入される前
     * @internal
     */
    readonly T_NEXT_DOM_BEFORE = 630;
    /**
     * 次のページのDOMにアクセス可能な状態
     * @internal
     */
    readonly T_BOTH_DOM_EXIST = 640;
    /**
     * 次のページのOBがアクセス可能な状態
     * @internal
     */
    readonly T_BOTH_OB_EXIST = 650;
    /**
     * トランジション終了時
     * @internal
     */
    readonly T_END = 660;
    /**
     * トランジションでエラーが出た時
     * @internal
     */
    readonly T_ERROR = 670;
    /**
     * ライブラリ内でエラーが発生した時
     * @internal
     */
    readonly ERROR = 9999;
}
declare const hookTypes: HookType;
type HookArgsMap = {
    [hookTypes.BEFORE_PAGE_INIT]: Negl;
    [hookTypes.MOUSEMOVE]: Mouse;
    [hookTypes.MOUSE_MESH_ENTER]: {
        mesh: Object3D;
        mouse: Mouse;
        ob?: Ob;
        meshUI: MeshUI;
    };
    [hookTypes.MOUSE_MESH_HOVER]: {
        mesh: Object3D;
        mouse: Mouse;
        ob?: Ob;
        meshUI: MeshUI;
    };
    [hookTypes.MOUSE_MESH_LEAVE]: {
        mesh: Object3D;
        mouse: Mouse;
        ob?: Ob;
        meshUI: MeshUI;
    };
    [hookTypes.MOUSE_MESH_CLICK]: {
        mesh: Object3D;
        mouse: Mouse;
        ob?: Ob;
        meshUI: MeshUI;
    };
    [hookTypes.SCROLL]: Scroller;
    [hookTypes.RENDER]: World;
    [hookTypes.RESIZE]: Viewport;
    [hookTypes.RESIZE_DELAY]: Viewport;
    [hookTypes.T_INIT]: {};
    [hookTypes.T_BEGIN]: TransitionState;
    [hookTypes.T_NEXT_DOM_BEFORE]: TransitionState;
    [hookTypes.T_BOTH_DOM_EXIST]: TransitionState;
    [hookTypes.T_BOTH_OB_EXIST]: TransitionState;
    [hookTypes.T_END]: TransitionState;
    [hookTypes.T_ERROR]: TransitionState;
    [hookTypes.ERROR]: Error;
};
/**
 * フックの{@link Hook#on 登録}、{@link Hook#off 削除}を行います。
 *
 * {@link hook} 変数として negl からインポートして使用することができます。
 *
 * **取得方法**
 * ```ts
 * import { hook } from "../negl";
 * // or
 * window.negl.hook
 * ```
 *
 * @since 1.0.0
 */
declare class Hook extends HookType {
    #private;
    constructor();
    checkHookExists(hook: number): void;
    /**
     * フックの実行を行います。各引数の詳細は {@link Hook.on} を参照してください。
     * @param hook フックタイプ
     * @param state フックに登録したコールバック関数の引数に渡る値
     * @param option フックオプション
     */
    trigger<T extends keyof HookArgsMap>(hook: T, state: HookArgsMap[T], option?: HookOptions): Promise<void>;
    /**
     * フックに紐づけたい(コールバック)関数の登録を行います。
     * フックに紐づけられた関数は特定の操作、またはイベントが発生したタイミングで実行されます。
     * また、フックを削除したい場合は {@link Hook#off} を使用してください。
     *
     * #### hook
     * フックタイプの種類は以下の通りです。
     *
     * | フックタイプ       | 発火タイミング                                 | コールバックの引数                                                                     |
     * | ----------------- | ---------------------------------------------- | ------------------------------------------------------------------------ |
     * | BEFORE_PAGE_INIT  | {@link Page} モジュールのデフォルト関数実行前                   | {@link Negl}                                                            |
     * | MOUSEMOVE         | マウスが動いている間連続的に発火                  | {@link mouse}                                                            |
     * | MOUSE_MESH_ENTER  | マウスがメッシュに入った時                     | { [mesh](https://threejs.org/docs/#api/en/objects/Mesh), {@link mouse}, {@link Ob ob}, {@link MeshUI meshUI} } |      |
     * | MOUSE_MESH_HOVER  | メッシュにホバーしている時                     | { [mesh](https://threejs.org/docs/#api/en/objects/Mesh), {@link mouse}, {@link Ob ob}, {@link MeshUI meshUI} } |      |
     * | MOUSE_MESH_LEAVE  | マウスがメッシュから出た時                     | { [mesh](https://threejs.org/docs/#api/en/objects/Mesh), {@link mouse}, {@link Ob ob}, {@link MeshUI meshUI} } |      |
     * | MOUSE_MESH_CLICK  | マウスがメッシュをクリックした時               | { [mesh](https://threejs.org/docs/#api/en/objects/Mesh), {@link mouse}, {@link Ob ob}, {@link MeshUI meshUI} } |      |
     * | SCROLL            | スクロールしている時                           | {@link scroller}                                                         |
     * | RENDER            | 毎フレーム（requestAnimationFrame）            | {@link world}                                                            |
     * | RESIZE            | 画面サイズが変わっている間連続的に発火         | {@link viewport}                                                         |
     * | RESIZE_DELAY      | 画面サイズの更新が終了した時                   | {@link viewport}                                                         |
     * | T_INIT            | ページトランジションの初期化時                 | {}*空のオブジェクト                                                        |
     * | T_BEGIN           | ページトランジションの開始時                   | {@link TransitionState state}                                                  |
     * | T_NEXT_DOM_BEFORE | 次のページのDOMが挿入される前                  | {@link TransitionState state}                                                  |
     * | T_BOTH_DOM_EXIST  | 次のページのDOMにアクセス可能な状態            | {@link TransitionState state}                                                  |
     * | T_BOTH_OB_EXIST   | 次のページのObオブジェクトがアクセス可能な状態 | {@link TransitionState state}                                                  |
     * | T_END             | ページトランジション終了時                     | {@link TransitionState state}                                                  |
     * | T_ERROR           | ページトランジションでエラーが出た時           | {@link TransitionState state}                                                  |
     * | ERROR             | negl 内でエラーが発生した時                    | Errorオブジェクト                                                         |
     *
     * #### callback
     * フックに登録したコールバック関数がイベント発火のタイミングで実行されます。
     *
     * #### option
     * オプションはオブジェクトで渡します。以下のプロパティが設定可能です。
     *
     * | プロパティ | 型      | 初期値 | 役割                                                                                                 |
     * | ---------- | ------- | ------ | ---------------------------------------------------------------------------------------------------- |
     * | type       | string  | base   | トランジションのタイプを設定します。（T_*から始まるフックでのみ使用）                                |
     * | priority   | number  | 0      | 同じ種類のフックに複数のコールバックが登録された場合に、priorityが大きい順番でフックが実行されます。 |
     * | once       | boolean | false  | trueの場合、一度実行されたフックは自動的に削除されます。                                             |
     *
     * **使用例**
     * ```ts copy
     * import { hook } from "../negl";
     *
     * // マウスがメッシュに入った時に一回発火（
     * hook.on(hook.BEFORE_PAGE_INIT, (negl) => {
     *    console.log(negl); // neglをログに出力
     * })
     * // マウスがメッシュに入った時に一回発火（HTMLタグに`data-interactive`を要付与）
     * hook.on(hook.MOUSE_MESH_ENTER, ({ mouse }) => {
     *    console.log(mouse.getClipPos()); // マウス位置をクリップ座標でログに出力
     * })
     *
     * // マウスがメッシュに入った時に一回発火（HTMLタグに`data-interactive`を要付与）
     * hook.on(hook.MOUSE_MESH_ENTER, ({ mesh }) => {
     *    mesh.scale.set(2,2,1); // マウスが重なった `mesh` をx,y軸方向に2倍に拡大
     * })
     *
     * // マウスがメッシュから出た時に一回発火（HTMLタグに`data-interactive`を要付与）
     * hook.on(hook.MOUSE_MESH_LEAVE, ({ mesh }) => {
     *    mesh.scale.set(1,1,1); // マウスが出た `mesh` を元の大きさに戻す
     * });
     *
     * // マウスをクリックした時に発火（HTMLタグに`data-interactive`を要付与）
     * hook.on(hook.MOUSE_MESH_CLICK, ({ mesh }) => {
     *    // クリックしたメッシュを右方向に 100 移動
     *    mesh.position.x += 100;
     * });
     *
     * // 注意） MOUSE_MESH_* のフックはHTMLタグに`data-interactive`を付与してください。
     * // <img data-webgl="some-effect" data-tex-1="/example.jpg" data-interactive />
     *
     * // スクロール中に発火
     * hook.on(hook.SCROLL, scroller => {
     *    const { delta } = scroller;
     *    console.log(delta.x, delta.y); // スクロールの変化量をログに出力
     * });
     *
     * // 毎フレームの処理
     * hook.on(hook.RENDER, world => {
     *    // 注意）`requestAnimationFrame` のタイミングで連続的に実行されます。
     *    // 重い処理は書かないでください。
     *    console.log(world.tick); // tickのカウントをログに表示
     * });
     *
     * // 画面リサイズ時に連続的に発火
     * hook.on(hook.RESIZE, viewport => {
     *    console.log(viewport.width); // canvasの横幅を取得
     * });
     *
     * // 画面リサイズの終了時に発火
     * hook.on(hook.RESIZE_DELAY, viewport => {
     *    console.log(viewport.width); // canvasの横幅を取得
     * });
     * ```
     * ページトランジション(T_*)に関するフックの使い方については {@link transition} を参照してください。
     */
    on<T extends keyof HookArgsMap>(hook: T, callback: HookCallback<HookArgsMap[T]>, option?: HookOptions): void;
    /**
     * {@link Hook#on}でフックに登録したコールバックを削除します。
     *
     * ```ts
     * // コールバック関数
     * function logTick(world) {
     *   console.log(world.tick);
     * }
     *
     * // hook.RENDERのフックにコールバック（logTick）を登録
     * hook.on(hook.RENDER, logTick);
     *
     * setTimeout(() => {
     *   // hook.RENDERのフックからコールバック（logTick）を削除
     *   hook.off(hook.RENDER, logTick);
     * }, 3000); // 3秒後
     *
     * // 3秒間`tick`の値がログに出力される。
     * ```
     */
    off<T extends keyof HookArgsMap>(hook: T, callback: HookCallback<HookArgsMap[T]>, option?: HookOptions): void;
}
/**
 * 特定のイベントに対する処理の{@link Hook#on 登録}、{@link Hook#off 削除}を制御する{@link Hook}クラスのオブジェクトです。
 * イベントが発生した際に登録したコールバック関数が実行されます。
 *
 * **取得方法**
 * ```ts copy
 * import { hook } from "../negl";
 * // or
 * window.negl.hook
 * ```
 *
 * 詳細は{@link Hook#on}を参照してください。
 */
declare const hook: Hook;
export { hook, Hook };
export default hook;
//# sourceMappingURL=hook.d.ts.map