﻿#pragma glslify: coverUv = require(../shader-util/coverUv)
#pragma glslify: rotate3d = require(glsl-rotate/rotation-3d)

uniform vec4 uResolution;
uniform vec2 uMouse;
uniform float uHover;
uniform float uTick;
// uniform int uLoop; // 2023/5/5 WebGL1.0対応 uLoopはシェーダ内で定数で定義に変更
const int uLoop = 15;
uniform float uProgress;

uniform sampler2D tex1;
varying vec2 vUv;

// 引数として受け取ったベクトル(v)を任意の回転軸(axis)に沿って回転(angle)させる関数。回転後のベクトルを返す。
// A function that rotates a given vector (v) around an arbitrary axis (axis) by an angle (angle). Returns the vector after rotation.
vec3 rotate(vec3 v, vec3 axis, float angle) {
  // rotate3dは第一引数に回転軸ベクトル、第二引数に回転角を取る
  // rotate3d takes the rotation axis vector as the first argument and the rotation angle as the second.
  mat4 matR = rotate3d(axis, angle);
  return (matR * vec4(v, 1.0)).xyz;
}

/*
* ２つの距離関数をスムーズに結合する関数。a,bは距離関数、kは係数
* ２つの関数から微分可能になるような関数を作る。（右微分係数と左微分係数が一致するような関数を作れば良い）。
* A function that smoothly blends two distance functions. a and b are distance functions, k is a coefficient.
* Creates a function that is differentiable from the two functions. (A function where the right and left derivative coefficients match is sufficient).
*/
float smin(float a, float b, float k) {
  float h = clamp(0.5 + 0.5 * (b - a) / k, 0.0, 1.0);
  return mix(b, a, h) - k * h * (1.0 - h);
}

// 環境マップを返す関数, eyeは光線ray、normalはSDFの法線ベクトル
// A function that returns an environment map, where eye is the ray direction, and normal is the normal vector of the SDF.
vec2 getmatcap(vec3 eye, vec3 normal) {
  vec3 reflected = reflect(eye, normal);
  float m = 2.8284271247461903 * sqrt(reflected.z + 1.0);
  return reflected.xy / m + 0.5;
}

// 球のSDF。半径r、中心が空間の原点
// The SDF of a sphere. Radius r, with the center at the origin of the space.
float sphereSDF(vec3 p, float r) {
  return length(p) - r;
}

// 立方体のSDF。
// The SDF of a cube.
float boxSDF(vec3 p, vec3 b) {
  vec3 q = abs(p) - b;
  return length(max(q, 0.0)) + min(max(q.x, max(q.y, q.z)), 0.0);
}

float rand(vec2 co) {
  return fract(sin(dot(co.xy, vec2(12.9898, 78.233))) * 43758.5453);
}

// 空間全体のSDF
// The SDF of the entire space.
float sceneSDF(vec3 p) {
  vec3 pRotated = rotate(p, vec3(1.), uTick / 50.); // 立法体の回転
  float box = boxSDF(pRotated, vec3(0.2));

  float final = box;
  // 中心のオブジェに向かってランダムに球が吸い込まれる
  // Spheres are randomly drawn towards the center object."
  for(int i = 0; i < uLoop; i++) {
    float randOffset = rand(vec2(i, 0.2));
    float progress = 1. - fract(uTick / 100. + randOffset * 3.);
    vec3 pos = vec3(sin(randOffset * 2. * PI), cos(randOffset * 2. * PI), 0.);
    float sphere = sphereSDF(p - pos * progress, 0.1 * sin(PI * progress));
    final = smin(final, sphere, 0.3);
  }

  return final;
}

// 点pにおける、SDFの等値面との法線ベクトルを求める関数。
// A function that calculates the normal vector at point p with respect to the isosurface of the SDF.
vec3 gradSDF(vec3 p) {
  float eps = 0.001; // 微小変化量
  return normalize(vec3(sceneSDF(p + vec3(eps, 0.0, 0.0)) - sceneSDF(p - vec3(eps, 0.0, 0.0)), // x成分の偏微分
  sceneSDF(p + vec3(0.0, eps, 0.0)) - sceneSDF(p - vec3(0.0, eps, 0.0)), // y成分の偏微分
  sceneSDF(p + vec3(0.0, 0.0, eps)) - sceneSDF(p - vec3(0.0, 0.0, eps))  // z成分の偏微分
  ));
}

void main() {
  vec2 newUV = coverUv(vUv, uResolution);
  newUV = (newUV - .5) * 2.;

  // カメラ（視点）の位置
  // The position of the camera (viewpoint).
  vec3 cPos = vec3(0.0, 0.0, 1.0);

  // 光源の位置
  // The position of the light source.
  vec3 lPos = vec3(2.0);

  // 光線の向きのベクトルを正規化。カメラは常にZ軸マイナス方向に向ける。
  // Normalize the direction vector of the ray. The camera always faces in the negative Z-axis direction.
  vec3 rPos = cPos; // 初期の光線の位置はカメラの位置.  The initial position of the ray is the position of the camera.

  gl_FragColor = vec4(0.);
  for(int i = 0; i < uLoop; i++) {
    if(sceneSDF(rPos) > 0.001) {
      rPos += sceneSDF(rPos) * ray;
    } else {
      float amb = 0.5; // 環境光の強さ. The strength of the ambient light.

      /*
      * 拡散光の計算。光線の位置（物体にヒットしたポイント）から光源に伸びるベクトルとSDFの法線ベクトルとの内積を計算する。
      * 内積がマイナスになる（角度が180度以上になる場合）場合は0にする。
      * Calculation of the diffuse light. Calculate the dot product between the vector extending from the position of the ray (the point where it hits the object) to the light source and the normal vector of the SDF.
      * If the dot product is negative (when the angle is more than 180 degrees), set it to 0.
      */
      vec3 sdfNormal = gradSDF(rPos);
      float diff = 0.9 * max(dot(normalize(lPos - rPos), sdfNormal), 0.0);

      // スフィア環境マップ作成
      // Creating a sphere environment map.
      vec2 matcapUV = getmatcap(ray, sdfNormal);
      vec3 color = texture2D(tex1, matcapUV).rgb;

      color *= diff + amb;

      gl_FragColor = vec4(color, uProgress);

      break;
    }
  }
}
