﻿#pragma glslify: coverUv = require(../shader-util/coverUv)
#pragma glslify: rotate3d = require(glsl-rotate/rotation-3d)

uniform vec4 uResolution;
uniform vec2 uMouse;
uniform float uHover;
uniform float uTick;
// uniform int uLoop; // 2023/5/5 WebGL1.0対応 uLoopはシェーダ内で定数で定義に変更
// For compatibility with WebGL1.0, uLoop is now defined as a constant within the shader.
const int uLoop = 15;
uniform float uProgress;

uniform sampler2D tex1;
varying vec2 vUv;

// 引数として受け取ったベクトル(v)を任意の回転軸(axis)に沿って回転(angle)させる関数。回転後のベクトルを返す。
// A function that rotates a given vector (v) around an arbitrary axis (axis) by an angle (angle). Returns the vector after rotation.
vec3 rotate(vec3 v, vec3 axis, float angle) {
  // rotate3dは第一引数に回転軸ベクトル、第二引数に回転角を取る
  // rotate3d takes the rotation axis vector as the first argument and the rotation angle as the second.
  mat4 matR = rotate3d(axis, angle);


  // vはアフィン変換するために4次元にする。
  // v is made 4-dimensional for affine transformation.
  return (matR * vec4(v, 1.0)).xyz;
}

/*
* ２つの距離関数をスムーズに結合する関数。a,bは距離関数、kは係数
* ２つの関数から微分可能になるような関数を作る。（右微分係数と左微分係数が一致するような関数を作れば良い）。
* A function that smoothly blends two distance functions. a and b are distance functions, k is a coefficient.
* Creates a function that is differentiable from the two functions. (A function where the right and left derivative coefficients match is sufficient).
*/
float smin(float a, float b, float k) {
  float h = clamp(0.5 + 0.5 * (b - a) / k, 0.0, 1.0);
  return mix(b, a, h) - k * h * (1.0 - h);
}

// 環境マップを返す関数, eyeは光線ray、normalはSDFの法線ベクトル
// A function that returns an environment map, where eye is the ray direction, and normal is the normal vector of the SDF.
vec2 getmatcap(vec3 eye, vec3 normal) {
  vec3 reflected = reflect(eye, normal);
  float m = 2.8284271247461903 * sqrt(reflected.z + 1.0);
  return reflected.xy / m + 0.5;
}

// 球のSDF。半径r、中心が空間の原点
// The SDF of a sphere. Radius r, with the center at the origin of the space.
float sphereSDF(vec3 p, float r) {
  return length(p) - r;
}

// 立方体のSDF。
// The SDF of a cube.
float boxSDF(vec3 p, vec3 b) {
  vec3 q = abs(p) - b;
  return length(max(q, 0.0)) + min(max(q.x, max(q.y, q.z)), 0.0);
}

// Box Frame - exact   (https://www.shadertoy.com/view/3ljcRh)
float sdBoxFrame(vec3 p, vec3 b, float e) {
  p = abs(p) - b;
  vec3 q = abs(p + e) - e;
  return min(min(length(max(vec3(p.x, q.y, q.z), 0.0)) + min(max(p.x, max(q.y, q.z)), 0.0), length(max(vec3(q.x, p.y, q.z), 0.0)) + min(max(q.x, max(p.y, q.z)), 0.0)), length(max(vec3(q.x, q.y, p.z), 0.0)) + min(max(q.x, max(q.y, p.z)), 0.0));
}

// 正八面体のSDF
// The SDF of an octahedron.
float octaSDF(vec3 p, float s) {
  p = abs(p);
  float m = p.x + p.y + p.z - s;
  vec3 q;
  if(3.0 * p.x < m) {
    q = p.xyz;
  } else if(3.0 * p.y < m) {

    q = p.yzx;
  } else if(3.0 * p.z < m) {

    q = p.zxy;
  } else {

    return m * 0.57735027;
  }

  float k = clamp(0.5 * (q.z - q.y + s), 0.0, s);
  return length(vec3(q.x, q.y - s + k, q.z - k));
}

// 空間全体のSDF
// The SDF of the entire space.
float sceneSDF(vec3 p) {
  vec3 pRotated = rotate(p, vec3(1.0), uTick / 200.);
  vec3 pAxisYRotete = rotate(pRotated, vec3(0.0, 1.0, 0.0), uTick / 200.);

  // マウスに追従する球
  // A sphere that follows the mouse.
  float sphereMouse = sphereSDF(p - 2.5 * vec3(uMouse * uHover - vec2(0.5) * uHover, 0.0), 0.5);

  float octa = octaSDF(pAxisYRotete, 1.2);
  // float box = sdBoxFrame(pAxisYRotete, vec3(0.6), 0.05);
  float box = boxSDF(pAxisYRotete, vec3(0.6));
  float mixed = mix(box, octa, 0.);

  float final = smin(sphereMouse, mixed, 0.8);

  return final;
}

// 点pにおける、SDFの等値面との法線ベクトルを求める関数。
// A function that calculates the normal vector at point p with respect to the isosurface of the SDF.
vec3 gradSDF(vec3 p) {
  float eps = 0.001; // 微小変化量 A small change amount.
  return normalize(vec3(sceneSDF(p + vec3(eps, 0.0, 0.0)) - sceneSDF(p - vec3(eps, 0.0, 0.0)), // x成分の偏微分. Partial derivative of the x component.
  sceneSDF(p + vec3(0.0, eps, 0.0)) - sceneSDF(p - vec3(0.0, eps, 0.0)), // y成分の偏微分
  sceneSDF(p + vec3(0.0, 0.0, eps)) - sceneSDF(p - vec3(0.0, 0.0, eps))  // z成分の偏微分
  ));
}

void main() {
  vec2 newUV = coverUv(vUv, uResolution);
  newUV = (newUV - .5) * 2.;

  // カメラ（視点）の位置
  // The position of the camera (viewpoint).
  vec3 cPos = vec3(0.0, 0.0, 2.0);

  // 光源の位置
  // The position of the light source.
  vec3 lPos = vec3(2.0);

  // 光線の向きのベクトルを正規化。カメラは常にZ軸マイナス方向に向ける。
  // Normalize the direction vector of the ray. The camera always faces in the negative Z-axis direction.
  vec3 ray = normalize(vec3(newUV, -1.));
  vec3 rPos = cPos; // 初期の光線の位置はカメラの位置. The initial position of the ray is the position of the camera.

  gl_FragColor = vec4(0.);
  for(int i = 0; i < uLoop; i++) {
    if(sceneSDF(rPos) > 0.001) {
      rPos += sceneSDF(rPos) * ray;
    } else {
      float amb = 0.5; // 環境光の強さ. The strength of the ambient light.

      /*
      * 拡散光の計算。光線の位置（物体にヒットしたポイント）から光源に伸びるベクトルとSDFの法線ベクトルとの内積を計算する。
      * 内積がマイナスになる（角度が180度以上になる場合）場合は0にする。
      * Calculation of the diffuse light. Calculate the dot product between the vector extending from the position of the ray (the point where it hits the object) to the light source and the normal vector of the SDF.
      * If the dot product is negative (when the angle is more than 180 degrees), set it to 0.
      */
      vec3 sdfNormal = gradSDF(rPos);
      float diff = 0.9 * max(dot(normalize(lPos - rPos), sdfNormal), 0.0);

      // スフィア環境マップ作成
      // Creating a sphere environment map.
      vec2 matcapUV = getmatcap(ray, sdfNormal);
      vec3 color = texture2D(tex1, matcapUV).rgb;

      color *= diff + amb;

      gl_FragColor = vec4(color, uProgress);

      break;
    }
  }
}