import {
  Slider,
  SlideIdx,
  SlideTransitionFn,
} from "#/parts/helper/slider/Slider";
import { config, utils, ObConstructorArgs } from "negl";
import { VideoTexture, ShaderMaterial } from "three";
import lilGUI from "lil-gui";

type Uniforms = ShaderMaterial["uniforms"];

/**
 * 一つのメッシュ上で複数テクスチャを切り替えて表示するスライダの基底クラス
 * 例）slider-frag、slider-noise、slider-displace
 *
 * Base class for a slider that switches between multiple textures on a single mesh.
 * Examples: slider-frag, slider-noise, slider-displace
 */
class SingleMeshSlider extends Slider {
  /**
   * スライダ切り替えアニメーション用関数を保持します。
   * Holds the function for slider transition animations.
   */
  #slideTransition: SlideTransitionFn;

  constructor(args: ObConstructorArgs) {
    super(args);

    this.#slideTransition = (uProgress, onStart, onComplete) => {
      onStart();
      utils.tween(uProgress, {
        value: 1,
        duration: 1,
        onComplete,
      });
    };
  }

  beforeCreateMesh() {
    this.activeSlideIdx = 0;
    this.running = false;
    this.playingVideo = null;
  }

  setupTexes(uniforms: Uniforms) {
    uniforms.texCurrent = { value: this.texes.get(`${config.prefix.tex}1`) };
    uniforms.texNext = { value: uniforms.texCurrent.value };
    return uniforms;
  }

  goTo(idx: SlideIdx) {
    this.activeSlideIdx = idx;

    if (this.running) return;
    this.running = true;

    this.uniforms.texNext.value = this.getNextTex(idx);

    this.#slideTransition(
      this.uniforms.uProgress,
      this.onStart.bind(this),
      this.onComplete.bind(this, idx)
    );
  }

  /**
   * スライダアニメーションをコールバック関数で変更します。
   * 次のようにして使用します。
   * Changes the slider animation with a callback function.
   * Use it as follows:
   *
   * ```js
   * const particle = world.getObByEl('.slider');
   * particle.changeSlideTransition((uProgress, onStart, onComplete) => {
   *   onStart();
   *   gsap.to(uProgress, {
   *     value: 1,
   *     duration: 5,
   *     ease: "power.inout",
   *     onComplete
   *   });
   * });
   * ```
   *
   * @param fn
   */
  changeSlideTransition(fn: SlideTransitionFn) {
    this.#slideTransition = fn;
  }

  /**
   * 次のスライダを表示します。
   * Displays the next slider.
   */
  goNext() {
    this.goTo(this.activeSlideIdx + 1);
  }

  /**
   * 前のスライダを表示します。
   * Displays the previous slider.
   */
  goPrev() {
    this.goTo(this.activeSlideIdx - 1);
  }

  /**
   * idx をスライド枚数の範囲内の数値で返却します。
   * Returns the index within the range of the number of slides.
   * @param idx
   * @returns
   */
  getModIdx(idx: SlideIdx) {
    idx = idx ?? this.activeSlideIdx;
    return ((idx % this.texes.size) + this.texes.size) % this.texes.size;
  }

  /**
   * 次に表示するテクスチャを取得します。
   * Get the texture to be displayed next.
   * @param idx
   * @returns
   */
  getNextTex(idx: SlideIdx) {
    const _idx = this.getModIdx(idx) + 1;
    const nextTex = this.texes.get(config.prefix.tex + _idx);
    return nextTex;
  }

  /**
   * スライダ切り替えアニメーション開始時の処理
   * Processing at the start of the slider transition animation
   */
  onStart() {}

  /**
   * スライダ切り替えアニメーション終了時の処理
   * Processing at the end of the slider transition animation
   */
  onComplete(idx: SlideIdx) {
    this.uniforms.texCurrent.value = this.uniforms.texNext.value;
    this.uniforms.uProgress.value = 0;
    this.playVideo();
    this.running = false;
  }

  async playVideo() {
    this.playingVideo?.pause?.();
    const texCurrentValue = this.uniforms.texCurrent.value;
    if (texCurrentValue instanceof VideoTexture) {
      this.playingVideo = texCurrentValue.source.data;

      this.playInterval = setInterval(() => {
        this.playingVideo!.play?.()
          .then(() => {
            if (!this.playingVideo!.paused) {
              clearInterval(this.playInterval);
            }
          })
          .catch(() => {});
      }, 50);
    }
  }

  afterInit() {
    const idx = this.activeSlideIdx;
    this.uniforms.texNext.value = this.getNextTex(idx);
    this.onComplete(idx);
  }

  debug(folder: lilGUI) {
    folder
      .add(this.uniforms.uProgress, "value", 0, 1, 0.1)
      .name("progress")
      .listen();
    const sliderIdx = { value: 0 };
    folder
      .add(sliderIdx, "value", -12, 12, 1)
      .name("goTo")
      .listen()
      .onChange(() => {
        this.goTo(sliderIdx.value);
      });
  }

  /**
   * クリックされた際の data-click-* の番号を返します。
   * Returns the number of data-click-* when clicked.
   * @param idx
   * @returns 実行したい data-click-* の番号(The number of data-click-* to execute)
   */
  getClickIdx(): number {
    return this.getModIdx(this.activeSlideIdx) + 1;
  }
}

export { SingleMeshSlider };
