import { SlideIdx, SlideTransitionFn, Slider } from "./Slider";
import { utils, config, INode, ObConstructorArgs } from "negl";
import { Points, PlaneGeometry, ShaderMaterial } from "three";
import lilGUI from "lil-gui";

type Uniforms = ShaderMaterial["uniforms"];

/**
 * 粒子（Particle）系のスライダの基底クラス  
 * 
 * スライドが切り替わるアニメーション時のみメッシュが表示され、スライダが切り替わった後は HTMLの画像や動画が表示され、メッシュは非表示になります。
 * 
 * 粒子系のスライダはこちらのクラスを継承して作成すると実装の手間が省けます。
 * 
 * 例）slider-particle-curl、slider-particle-z など。
 */
class ParticleSlider extends Slider {
  /**
   * スライダ切り替えアニメーション用関数を保持します。
   */
  #slideTransition: SlideTransitionFn;

  constructor(args: ObConstructorArgs) {
    super(args);

    this.#slideTransition = (uProgress, onStart, onComplete) => {
      onStart();
      utils.tween(uProgress, {
        value: 1,
        duration: 1,
        onComplete,
      });
    };
  }

  beforeCreateMesh() {
    this.activeSlideIdx = 0;
    this.running = false;
    this.playingVideo = null;
    this.$.childMediaEls = [];
    this.pointSize = utils.isTouchDevices() ? 1 : 2;
    this.pointInterval = 4;
    this.clsName = "particle-child";
  }

  setupGeometry() {
    const width = this.rect.width,
      height = this.rect.height,
      wSeg = Math.floor(width / this.pointInterval),
      hSeg = Math.floor(height / this.pointInterval);

    const plane = new PlaneGeometry(width, height, wSeg, hSeg);
    return plane;
  }

  setupTexes(uniforms: Uniforms) {
    uniforms.texCurrent = { value: this.texes.get(`${config.prefix.tex}1`) };
    uniforms.texNext = { value: uniforms.texCurrent.value };
    return uniforms;
  }

  setupMesh() {
    return new Points(this.geometry, this.material);
  }

  setupUniforms() {
    const uniforms = super.setupUniforms();
    uniforms.uPointSize = { value: this.pointSize };
    return uniforms;
  }

  goTo(idx: SlideIdx) {
    this.activeSlideIdx = idx;
    const _idx = this.getTexIdx(idx);

    if (this.running) return;
    this.running = true;

    const nextTex = this.texes.get(config.prefix.tex + _idx);
    this.uniforms.texNext.value = nextTex;

    this.#slideTransition(
      this.uniforms.uProgress,
      this.onStart.bind(this),
      this.onComplete.bind(this, idx)
    );
  }

  /**
   * スライダアニメーションをコールバック関数で変更します。  
   * 次のようにして使用します。
   * 
   * ```js 
   * const particle = world.getObByEl('.slider');
   * particle.changeSlideTransition((uProgress, onStart, onComplete) => {
   *   onStart();
   *   gsap.to(uProgress, {
   *     value: 1,
   *     duration: 5,
   *     ease: "power.inout",
   *     onComplete
   *   });
   * });
   * ```
   * 
   * @param fn 
   */
  changeSlideTransition(fn: SlideTransitionFn) {
    this.#slideTransition = fn;
  }

  /**
   * スライダアニメーション開始時の処理
   * @param idx 
   */
  onStart() {
    const children = this.$.childMediaEls as HTMLElement[];
    children.forEach((el) => {
      el.style.opacity = "0";
      el.style.pointerEvents = "none";
      if (el instanceof HTMLVideoElement) {
        el.pause();
      }
    });
    this.mesh.visible = true;
    this.uniforms.uAlpha.value = 1;
  }

  /**
   * スライダアニメーション終了時の処理
   * @param idx 
   */
  onComplete(idx: SlideIdx) {
    this.uniforms.texCurrent.value = this.uniforms.texNext.value;
    this.uniforms.uProgress.value = 0;
    const activeEl = this.getChildMediaEl(idx);
    activeEl.style.opacity = "1";
    activeEl.style.pointerEvents = "";
    this.mesh.visible = false;
    this.uniforms.uAlpha.value = 0;
    this.running = false;
    if (
      activeEl instanceof HTMLVideoElement &&
      (activeEl.paused || utils.isSafari())
    ) {
      activeEl.play();
    }
  }

  /**
   * テクスチャのインデックスを取得
   * @param idx 
   * @returns 
   */
  getTexIdx(idx: SlideIdx) {
    return this.getModIdx(idx) + 1;
  }

  /**
   * HTML要素を取得
   * @param idx 
   * @returns 
   */
  getChildMediaEl(idx: SlideIdx): HTMLElement {
    const _idx = this.getModIdx(idx);
    const children = this.$.childMediaEls as HTMLElement[];
    return children[_idx];
  }

  afterInit() {
    let i = 0;

    this.texes.forEach((tex) => {
      const parent = this.$.el.parentElement!;
      const particleChildren = INode.qsAll(`.${this.clsName}`);
      let childEl = particleChildren[i];

      if (!childEl) {
        const mediaEl = tex!.source.data.cloneNode() as HTMLMediaElement;

        const dataVal = this.dataAttrs[`${config.prefix.click}-${i}`];

        /**
         * data-click-*が設定されている場合はリンクを挿入します。
         * なお、ParticleSliderの場合URL以外は設定できません。
         */
        if (dataVal) {
          const { href, target } = utils.getHrefObj(dataVal.split(","))!;
          const linkEl = INode.htmlToEl(
            `<a href="${href}" target="${target}"></a>`
          );
          linkEl.append(mediaEl);
          childEl = linkEl;
        } else {
          childEl = mediaEl;
        }

        childEl.classList.add(this.clsName);
        parent.append(childEl);
      }

      const children = this.$.childMediaEls as HTMLElement[];
      children.push(childEl as HTMLElement);

      i++;
    });

    this.onStart();
    this.onComplete(this.activeSlideIdx);
  }

  debug(folder: lilGUI) {
    folder
      .add(this.uniforms.uProgress, "value", 0, 1, 0.1)
      .name("progress")
      .listen();
    const sliderIdx = { value: 0 };
    folder
      .add(sliderIdx, "value", -12, 12, 1)
      .name("goTo")
      .listen()
      .onChange(() => {
        this.goTo(sliderIdx.value);
      });
  }
}

export { ParticleSlider };
