import { Vector4 } from "three";
import { ShaderPass } from "three/examples/jsm/postprocessing/ShaderPass";
import { getComposer } from "#/parts/helper/getComposer";

let streamPass;
export function getStreamPass() {
  return streamPass;
}

async function initStreamPass({ world, hook, gui }) {
  /**
   * 2回以上Passをシーンに追加しないため、すでに instance が設定されている場合はそれを返却
   * To avoid adding the Pass to the scene more than twice, return the existing instance if one is already set.
   */
  if (streamPass) return streamPass;
  // HTML要素をテキストで渡してObオブジェクトを作成
  const o = await world.createOb(`
    <div data-webgl="stream">
      <style>
        [data-webgl="stream"] {
          position: absolute;
          visibility: hidden;
          width: 1px;
          height: 1px;
        }
      </style>
    </div>
  `);
  const { material, uniforms } = o;
  material.uniforms = uniforms;
  // シーンにエフェクトを表示しないため、シーンから除外
  // To prevent displaying effects in the scene, exclude it from the scene.
  world.removeOb(o, false);
  uniforms.tDiffuse = { value: null };

  uniforms.uParam = { value: new Vector4(2, 2.7, 5.7, 7.2) };
  uniforms.uTick = { value: 0 };
  uniforms.uProgress.value = 0;
  uniforms.uAlpha.value = 1;
  uniforms.uScale.value = 1;

  material.alphaTest = 0;

  // 参考：[レクチャー] - https://learn.not-equal.dev/ja/lecture/ja_webgl_basic-060-070_shaderpass/
  // Reference：[Lecture] - https://learn.not-equal.dev/en/lecture/en_webgl_basic-060-070_shaderpass/
  const pass = new ShaderPass(material);
  const composer = getComposer();
  composer.addPass(pass);

  // 毎フレームでuTick.valueの更新
  // Updating uTick.value every frame.
  hook.on(hook.RENDER, ({ tick }) => {
    pass.uniforms.uTick.value = tick;
  });

  function addGUI() {
    gui.add((lilGUI) => {
      const folder = lilGUI.addFolder("stream");
      folder
        .add(pass.uniforms.uProgress, "value", 0, 1, 0.01)
        .name("progress")
        .listen();
      folder
        .add(pass.uniforms.uAlpha, "value", 0, 1, 0.01)
        .name("alpha")
        .listen();
      folder
        .add(pass.uniforms.uScale, "value", -10, 10, 0.01)
        .name("scale")
        .listen();
      folder
        .add(pass.uniforms.uParam.value, "x", 0, 20, 0.01)
        .name("noise.x")
        .listen();
      folder
        .add(pass.uniforms.uParam.value, "y", 0, 20, 0.01)
        .name("noise.y")
        .listen();
      folder
        .add(pass.uniforms.uParam.value, "z", 0, 20, 0.01)
        .name("noise.z")
        .listen();
      folder
        .add(pass.uniforms.uParam.value, "w", 0, 20, 0.01)
        .name("noise.w")
        .listen();
    });
  }
  addGUI();
  hook.on(hook.T_END, addGUI);

  function setProgress(value) {
    uniforms.uProgress.value = value;
  }
  function setScale(value) {
    uniforms.uScale.value = value;
  }
  function setAlpha(value) {
    uniforms.uAlpha.value = value;
  }
  function removePass(dispose = true) {
    composer.removePass(pass);
    if (dispose) {
      o.geometry.dispose();
      o.material.dispose();
    }
  }

  function addPass() {
    composer.addPass(pass);
  }

  streamPass = {
    addPass,
    removePass,
    setProgress,
    setScale,
    setAlpha,
  };

  return streamPass;
}

export default initStreamPass;
